<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Transaction;
use App\Models\VirtualCard;
use App\Models\CronLog;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;

class GenerateDailyReportsCommand extends Command
{
    protected $signature = 'fintech:generate-daily-reports';
    protected $description = 'Generate daily business reports';

    public function handle()
    {
        $startTime = microtime(true);
        $startMemory = memory_get_usage(true);
        
        try {
            $this->info('Starting daily report generation...');
            
            $yesterday = now()->subDay();
            $reportData = $this->generateReportData($yesterday);
            
            // Generate report content
            $reportContent = $this->generateReportContent($reportData, $yesterday);
            
            // Save report to storage
            $fileName = "daily-report-{$yesterday->format('Y-m-d')}.json";
            Storage::disk('local')->put("reports/daily/{$fileName}", json_encode($reportData, JSON_PRETTY_PRINT));
            
            // Generate human-readable report
            $htmlReport = $this->generateHtmlReport($reportData, $yesterday);
            $htmlFileName = "daily-report-{$yesterday->format('Y-m-d')}.html";
            Storage::disk('local')->put("reports/daily/{$htmlFileName}", $htmlReport);
            
            // Send report to admin email if configured
            $this->sendReportToAdmin($reportData, $yesterday);
            
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $message = "Generated daily report for {$yesterday->format('Y-m-d')}";
            $this->info($message);
            
            CronLog::create([
                'task_name' => 'generate_daily_reports',
                'status' => 'success',
                'message' => $message,
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage
            ]);
            
        } catch (\Exception $e) {
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $this->error('Daily report generation failed: ' . $e->getMessage());
            
            CronLog::create([
                'task_name' => 'generate_daily_reports',
                'status' => 'failed',
                'message' => 'Daily report generation failed',
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage,
                'error_details' => [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]
            ]);
        }
    }
    
    private function generateReportData($date)
    {
        $startOfDay = $date->startOfDay();
        $endOfDay = $date->endOfDay();
        
        // User metrics
        $userMetrics = [
            'new_registrations' => User::whereBetween('created_at', [$startOfDay, $endOfDay])->count(),
            'verified_users' => User::whereBetween('email_verified_at', [$startOfDay, $endOfDay])->count(),
            'kyc_completed' => User::where('kyc_status', 'verified')
                ->whereBetween('updated_at', [$startOfDay, $endOfDay])->count(),
            'total_users' => User::count(),
            'active_users' => User::whereHas('transactions', function($query) use ($startOfDay, $endOfDay) {
                $query->whereBetween('created_at', [$startOfDay, $endOfDay]);
            })->count()
        ];
        
        // Transaction metrics
        $transactionMetrics = [
            'total_transactions' => Transaction::whereBetween('created_at', [$startOfDay, $endOfDay])->count(),
            'successful_transactions' => Transaction::where('status', 'completed')
                ->whereBetween('created_at', [$startOfDay, $endOfDay])->count(),
            'failed_transactions' => Transaction::where('status', 'failed')
                ->whereBetween('created_at', [$startOfDay, $endOfDay])->count(),
            'pending_transactions' => Transaction::whereIn('status', ['pending', 'processing'])
                ->whereBetween('created_at', [$startOfDay, $endOfDay])->count(),
            'total_volume' => Transaction::where('status', 'completed')
                ->whereBetween('created_at', [$startOfDay, $endOfDay])
                ->sum('amount'),
            'total_fees' => Transaction::where('status', 'completed')
                ->whereBetween('created_at', [$startOfDay, $endOfDay])
                ->sum('fee')
        ];
        
        // Transaction by category
        $transactionsByCategory = Transaction::whereBetween('created_at', [$startOfDay, $endOfDay])
            ->selectRaw('category, COUNT(*) as count, SUM(amount) as volume')
            ->groupBy('category')
            ->get()
            ->keyBy('category');
        
        // Virtual card metrics
        $cardMetrics = [
            'new_cards' => VirtualCard::whereBetween('created_at', [$startOfDay, $endOfDay])->count(),
            'active_cards' => VirtualCard::where('status', 'active')->count(),
            'frozen_cards' => VirtualCard::where('status', 'frozen')->count(),
            'card_transactions' => Transaction::where('category', 'virtual_card_transaction')
                ->whereBetween('created_at', [$startOfDay, $endOfDay])->count()
        ];
        
        // Wallet metrics
        $walletMetrics = [
            'total_wallet_balance' => \App\Models\Wallet::sum('balance'),
            'total_locked_balance' => \App\Models\Wallet::sum('locked_balance'),
            'wallet_funding_volume' => Transaction::where('category', 'wallet_funding')
                ->where('status', 'completed')
                ->whereBetween('created_at', [$startOfDay, $endOfDay])
                ->sum('amount')
        ];
        
        // Bill payment metrics
        $billPaymentMetrics = [
            'airtime_purchases' => Transaction::where('category', 'airtime_purchase')
                ->whereBetween('created_at', [$startOfDay, $endOfDay])->count(),
            'data_purchases' => Transaction::where('category', 'data_purchase')
                ->whereBetween('created_at', [$startOfDay, $endOfDay])->count(),
            'electricity_payments' => Transaction::where('category', 'bill_payment')
                ->whereBetween('created_at', [$startOfDay, $endOfDay])->count(),
            'total_bill_volume' => Transaction::whereIn('category', ['airtime_purchase', 'data_purchase', 'bill_payment'])
                ->where('status', 'completed')
                ->whereBetween('created_at', [$startOfDay, $endOfDay])
                ->sum('amount')
        ];
        
        // Error metrics
        $errorMetrics = [
            'cron_errors' => CronLog::where('status', 'failed')
                ->whereBetween('executed_at', [$startOfDay, $endOfDay])->count(),
            'transaction_failures' => $transactionMetrics['failed_transactions'],
            'system_alerts' => 0 // Placeholder for system alerts
        ];
        
        return [
            'date' => $date->format('Y-m-d'),
            'users' => $userMetrics,
            'transactions' => $transactionMetrics,
            'transactions_by_category' => $transactionsByCategory,
            'virtual_cards' => $cardMetrics,
            'wallets' => $walletMetrics,
            'bill_payments' => $billPaymentMetrics,
            'errors' => $errorMetrics,
            'generated_at' => now()->toISOString()
        ];
    }
    
    private function generateReportContent($data, $date)
    {
        return "Daily Business Report for {$date->format('F j, Y')}\n\n" .
               "User Metrics:\n" .
               "- New Registrations: {$data['users']['new_registrations']}\n" .
               "- Active Users: {$data['users']['active_users']}\n" .
               "- KYC Completions: {$data['users']['kyc_completed']}\n\n" .
               "Transaction Metrics:\n" .
               "- Total Transactions: {$data['transactions']['total_transactions']}\n" .
               "- Successful: {$data['transactions']['successful_transactions']}\n" .
               "- Failed: {$data['transactions']['failed_transactions']}\n" .
               "- Total Volume: ₦" . number_format($data['transactions']['total_volume'], 2) . "\n" .
               "- Total Fees: ₦" . number_format($data['transactions']['total_fees'], 2) . "\n\n" .
               "Virtual Cards:\n" .
               "- New Cards: {$data['virtual_cards']['new_cards']}\n" .
               "- Active Cards: {$data['virtual_cards']['active_cards']}\n\n" .
               "Bill Payments:\n" .
               "- Airtime: {$data['bill_payments']['airtime_purchases']}\n" .
               "- Data: {$data['bill_payments']['data_purchases']}\n" .
               "- Electricity: {$data['bill_payments']['electricity_payments']}\n";
    }
    
    private function generateHtmlReport($data, $date)
    {
        return view('reports.daily', compact('data', 'date'))->render();
    }
    
    private function sendReportToAdmin($data, $date)
    {
        $adminEmail = config('app.admin_email');
        
        if ($adminEmail) {
            try {
                Mail::to($adminEmail)->send(new \App\Mail\DailyReportMail($data, $date));
                $this->info("Daily report sent to admin: {$adminEmail}");
            } catch (\Exception $e) {
                $this->warn("Failed to send daily report email: " . $e->getMessage());
            }
        }
    }
}