@extends('layouts.app')

@section('title', 'Virtual Cards')

@section('content')
<div class="mb-6">
    <div class="flex justify-between items-center">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Virtual Cards</h1>
            <p class="text-gray-600">Manage your virtual cards for online payments</p>
        </div>
        @if(auth()->user()->hasCompletedKyc())
        <button onclick="openCreateCardModal()" class="btn-primary">
            <span class="mr-2">+</span> Request New Card
        </button>
        @endif
    </div>
</div>

@if(!auth()->user()->hasCompletedKyc())
<div class="card bg-yellow-50 border-yellow-200 mb-6">
    <div class="flex items-start">
        <span class="text-2xl mr-3">⚠️</span>
        <div>
            <h3 class="text-lg font-bold text-yellow-800 mb-2">KYC Verification Required</h3>
            <p class="text-yellow-700 text-sm mb-4">
                Complete your identity verification to request virtual cards.
            </p>
            <a href="{{ route('kyc.index') }}" class="btn-primary">Complete KYC</a>
        </div>
    </div>
</div>
@endif

<!-- Card Statistics -->
<div class="grid md:grid-cols-4 gap-4 mb-6">
    <div class="bg-gradient-to-r from-blue-50 to-blue-100 rounded-lg p-4">
        <h3 class="text-sm font-medium text-blue-600 mb-1">Total Cards</h3>
        <p class="text-2xl font-bold text-blue-900">{{ $cards->total() }}</p>
    </div>
    
    <div class="bg-gradient-to-r from-green-50 to-green-100 rounded-lg p-4">
        <h3 class="text-sm font-medium text-green-600 mb-1">Active Cards</h3>
        <p class="text-2xl font-bold text-green-900">{{ $cards->where('status', 'active')->count() }}</p>
    </div>
    
    <div class="bg-gradient-to-r from-orange-50 to-orange-100 rounded-lg p-4">
        <h3 class="text-sm font-medium text-orange-600 mb-1">Frozen Cards</h3>
        <p class="text-2xl font-bold text-orange-900">{{ $cards->where('status', 'frozen')->count() }}</p>
    </div>
    
    <div class="bg-gradient-to-r from-purple-50 to-purple-100 rounded-lg p-4">
        <h3 class="text-sm font-medium text-purple-600 mb-1">Total Balance</h3>
        <p class="text-2xl font-bold text-purple-900">₦{{ number_format($cards->sum('balance'), 2) }}</p>
    </div>
</div>

<!-- Cards List -->
@if($cards && $cards->count() > 0)
    <div class="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
        @foreach($cards as $card)
        <div class="card p-0 overflow-hidden">
            <!-- Card Design -->
            <div class="bg-gradient-to-r {{ $card->status === 'active' ? 'from-gray-900 to-gray-700' : 'from-gray-500 to-gray-400' }} text-white p-6 relative">
                <div class="flex justify-between items-start mb-6">
                    <div>
                        <p class="text-xs opacity-75 uppercase tracking-wide">{{ $card->card_type }} Card</p>
                        <p class="text-sm opacity-90">{{ $card->currency }}</p>
                    </div>
                    <div class="text-right">
                        <span class="px-2 py-1 bg-{{ $card->status === 'active' ? 'green' : ($card->status === 'frozen' ? 'yellow' : 'red') }}-500 rounded text-xs">
                            {{ ucfirst($card->status) }}
                        </span>
                    </div>
                </div>
                
                <div class="mb-6">
                    <p class="text-xl font-mono tracking-wider">
                        {{ substr($card->card_number, 0, 4) }} **** **** {{ substr($card->card_number, -4) }}
                    </p>
                </div>
                
                <div class="flex justify-between items-end">
                    <div>
                        <p class="text-xs opacity-75">Balance</p>
                        <p class="text-lg font-bold">₦{{ number_format($card->balance, 2) }}</p>
                    </div>
                    <div class="text-right">
                        <p class="text-xs opacity-75">Limit</p>
                        <p class="text-sm">₦{{ number_format($card->spending_limit, 0) }}</p>
                    </div>
                </div>
                
                <!-- Card type indicator -->
                <div class="absolute top-4 right-4">
                    @if($card->card_type === 'international')
                        <span class="text-xs bg-blue-500 px-2 py-1 rounded">🌍 International</span>
                    @else
                        <span class="text-xs bg-green-500 px-2 py-1 rounded">🇳🇬 Domestic</span>
                    @endif
                </div>
            </div>
            
            <!-- Card Actions -->
            <div class="p-4">
                <div class="flex justify-between items-center mb-3">
                    <span class="text-sm text-gray-600">Created {{ $card->created_at->format('M d, Y') }}</span>
                    <span class="text-sm text-gray-600">Usage: {{ $card->transactions_count ?? 0 }}</span>
                </div>
                
                <!-- Quick Actions -->
                <div class="grid grid-cols-2 gap-2 mb-3">
                    @if($card->status === 'active')
                        <button onclick="topUpCard('{{ $card->id }}')" class="text-xs px-3 py-2 bg-green-100 text-green-800 rounded hover:bg-green-200">
                            💰 Top Up
                        </button>
                        <button onclick="freezeCard('{{ $card->id }}')" class="text-xs px-3 py-2 bg-yellow-100 text-yellow-800 rounded hover:bg-yellow-200">
                            ❄️ Freeze
                        </button>
                    @elseif($card->status === 'frozen')
                        <button onclick="topUpCard('{{ $card->id }}')" class="text-xs px-3 py-2 bg-green-100 text-green-800 rounded hover:bg-green-200">
                            💰 Top Up
                        </button>
                        <button onclick="unfreezeCard('{{ $card->id }}')" class="text-xs px-3 py-2 bg-blue-100 text-blue-800 rounded hover:bg-blue-200">
                            🔥 Unfreeze
                        </button>
                    @endif
                </div>
                
                <div class="grid grid-cols-2 gap-2">
                    <button onclick="viewCardDetails('{{ $card->id }}')" class="text-xs px-3 py-2 bg-gray-100 text-gray-800 rounded hover:bg-gray-200">
                        👁️ View Details
                    </button>
                    <button onclick="viewTransactions('{{ $card->id }}')" class="text-xs px-3 py-2 bg-blue-100 text-blue-800 rounded hover:bg-blue-200">
                        📊 Transactions
                    </button>
                </div>
            </div>
        </div>
        @endforeach
    </div>
    
    <!-- Pagination -->
    <div class="mt-6">
        {{ $cards->links() }}
    </div>
@else
    <div class="card text-center py-12">
        <span class="text-6xl mb-4 block">💳</span>
        <h3 class="text-xl font-bold text-gray-900 mb-2">No Virtual Cards Yet</h3>
        <p class="text-gray-600 mb-6">
            @if(auth()->user()->hasCompletedKyc())
                Request your first virtual card to start making secure online payments.
            @else
                Complete your KYC verification to request virtual cards.
            @endif
        </p>
        @if(auth()->user()->hasCompletedKyc())
        <button onclick="openCreateCardModal()" class="btn-primary">
            Request Your First Card
        </button>
        @else
        <a href="{{ route('kyc.index') }}" class="btn-primary">
            Complete KYC First
        </a>
        @endif
    </div>
@endif

<!-- Create Card Modal -->
<div id="createCardModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-medium text-gray-900">Request New Virtual Card</h3>
                <button onclick="closeCreateCardModal()" class="text-gray-400 hover:text-gray-600">
                    <span class="text-2xl">&times;</span>
                </button>
            </div>
            
            <form id="createCardForm" method="POST" action="{{ route('cards.create') }}">
                @csrf
                <div class="space-y-4">
                    <div>
                        <label for="card_type" class="block text-sm font-medium text-gray-700 mb-2">Card Type</label>
                        <select name="card_type" id="card_type" required class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="">Select Card Type</option>
                            <option value="domestic">Domestic (Nigeria only)</option>
                            <option value="international">International (Worldwide)</option>
                        </select>
                    </div>
                    
                    <div>
                        <label for="spending_limit" class="block text-sm font-medium text-gray-700 mb-2">Monthly Spending Limit</label>
                        <select name="spending_limit" id="spending_limit" required class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="">Select Limit</option>
                            <option value="50000">₦50,000</option>
                            <option value="100000">₦100,000</option>
                            <option value="500000">₦500,000</option>
                            <option value="1000000">₦1,000,000</option>
                        </select>
                    </div>
                    
                    <div>
                        <label for="card_pin" class="block text-sm font-medium text-gray-700 mb-2">4-Digit PIN</label>
                        <input type="password" name="pin" id="card_pin" maxlength="4" pattern="[0-9]{4}" required 
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                               placeholder="Enter 4-digit PIN">
                    </div>
                    
                    <div>
                        <label for="confirm_pin" class="block text-sm font-medium text-gray-700 mb-2">Confirm PIN</label>
                        <input type="password" name="confirm_pin" id="confirm_pin" maxlength="4" pattern="[0-9]{4}" required 
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                               placeholder="Confirm PIN">
                    </div>
                    
                    <div class="bg-blue-50 border border-blue-200 rounded-lg p-3">
                        <p class="text-sm text-blue-800">
                            <strong>Note:</strong> Card creation fee is ₦1,000. International cards have higher limits but additional fees may apply.
                        </p>
                    </div>
                </div>
                
                <div class="flex justify-center space-x-4 mt-6">
                    <button type="button" onclick="closeCreateCardModal()" 
                            class="px-4 py-2 bg-gray-300 text-gray-700 rounded-md hover:bg-gray-400">
                        Cancel
                    </button>
                    <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                        Create Card (₦1,000)
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Card Details Modal -->
<div id="cardDetailsModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-medium text-gray-900">Card Details</h3>
                <button onclick="closeCardDetailsModal()" class="text-gray-400 hover:text-gray-600">
                    <span class="text-2xl">&times;</span>
                </button>
            </div>
            <div id="cardDetailsContent">
                <!-- Card details will be loaded here -->
            </div>
        </div>
    </div>
</div>

<script>
function openCreateCardModal() {
    document.getElementById('createCardModal').classList.remove('hidden');
}

function closeCreateCardModal() {
    document.getElementById('createCardModal').classList.add('hidden');
}

function openCardDetailsModal() {
    document.getElementById('cardDetailsModal').classList.remove('hidden');
}

function closeCardDetailsModal() {
    document.getElementById('cardDetailsModal').classList.add('hidden');
}

function topUpCard(cardId) {
    const amount = prompt('Enter top-up amount (₦):');
    if (amount && !isNaN(amount) && parseFloat(amount) > 0) {
        // Implementation for card top-up
        alert(`Top-up of ₦${amount} initiated for card ${cardId}`);
    }
}

function freezeCard(cardId) {
    if (confirm('Are you sure you want to freeze this card? You can unfreeze it later.')) {
        // Make AJAX request to freeze card
        fetch(`{{ url('/cards') }}/${cardId}/freeze`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json',
            },
        })
        .then(response => response.json())
        .then(data => {
            alert(data.message || 'Card frozen successfully');
            location.reload();
        })
        .catch(error => {
            alert('Error freezing card');
        });
    }
}

function unfreezeCard(cardId) {
    if (confirm('Are you sure you want to unfreeze this card?')) {
        // Make AJAX request to unfreeze card
        fetch(`{{ url('/cards') }}/${cardId}/unfreeze`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json',
            },
        })
        .then(response => response.json())
        .then(data => {
            alert(data.message || 'Card unfrozen successfully');
            location.reload();
        })
        .catch(error => {
            alert('Error unfreezing card');
        });
    }
}

function viewCardDetails(cardId) {
    openCardDetailsModal();
    document.getElementById('cardDetailsContent').innerHTML = '<p class="text-center">Loading...</p>';
    
    // Load card details (placeholder implementation)
    setTimeout(() => {
        document.getElementById('cardDetailsContent').innerHTML = `
            <div class="space-y-3">
                <div class="bg-gray-50 rounded-lg p-3">
                    <p class="text-sm text-gray-600">Card Number</p>
                    <p class="font-mono text-lg">**** **** **** ****</p>
                    <button onclick="revealCardNumber('${cardId}')" class="text-blue-600 text-sm">Click to reveal</button>
                </div>
                <div class="bg-gray-50 rounded-lg p-3">
                    <p class="text-sm text-gray-600">CVV</p>
                    <p class="font-mono">***</p>
                    <button onclick="revealCVV('${cardId}')" class="text-blue-600 text-sm">Click to reveal</button>
                </div>
                <div class="bg-gray-50 rounded-lg p-3">
                    <p class="text-sm text-gray-600">Expiry Date</p>
                    <p class="font-mono">**/**</p>
                    <button onclick="revealExpiry('${cardId}')" class="text-blue-600 text-sm">Click to reveal</button>
                </div>
                <div class="mt-4 text-xs text-gray-500 text-center">
                    Card details are hidden for security. Click to reveal individual fields.
                </div>
            </div>
        `;
    }, 500);
}

function viewTransactions(cardId) {
    window.location.href = `{{ url('/transactions') }}?card_id=${cardId}`;
}

function revealCardNumber(cardId) {
    // Implementation for revealing card number with security confirmation
    alert('Security verification required to reveal card number');
}

function revealCVV(cardId) {
    // Implementation for revealing CVV
    alert('Security verification required to reveal CVV');
}

function revealExpiry(cardId) {
    // Implementation for revealing expiry date
    alert('Security verification required to reveal expiry date');
}

// Close modals when clicking outside
document.getElementById('createCardModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeCreateCardModal();
    }
});

document.getElementById('cardDetailsModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeCardDetailsModal();
    }
});

// Validate PIN confirmation
document.getElementById('confirm_pin').addEventListener('input', function() {
    const pin = document.getElementById('card_pin').value;
    const confirmPin = this.value;
    
    if (confirmPin && pin !== confirmPin) {
        this.setCustomValidity('PINs do not match');
    } else {
        this.setCustomValidity('');
    }
});
</script>
@endsection