<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Wallet;
use Illuminate\Foundation\Testing\RefreshDatabase;

class ApplicationIntegrityTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test user registration flow
     */
    public function test_user_registration_creates_user_and_wallet(): void
    {
        $userData = [
            'first_name' => 'Test',
            'last_name' => 'User',
            'email' => 'test@example.com',
            'phone' => '+234800123456',
            'password' => 'password123',
            'password_confirmation' => 'password123',
            'date_of_birth' => '1990-01-01',
            'gender' => 'male'
        ];

        $response = $this->post('/api/auth/register', $userData);

        $response->assertStatus(201);
        $response->assertJsonStructure([
            'success',
            'message',
            'data' => [
                'user',
                'token',
                'requires_phone_verification',
                'requires_kyc'
            ]
        ]);

        // Check user was created
        $this->assertDatabaseHas('users', [
            'email' => 'test@example.com',
            'phone' => '+234800123456'
        ]);

        // Check wallet was created
        $user = User::where('email', 'test@example.com')->first();
        $this->assertDatabaseHas('wallets', [
            'user_id' => $user->id,
            'currency' => 'NGN'
        ]);
    }

    /**
     * Test API authentication endpoints
     */
    public function test_api_authentication_endpoints_exist(): void
    {
        // Registration endpoint
        $response = $this->post('/api/auth/register');
        $this->assertTrue(in_array($response->status(), [422, 201])); // Should not be 404

        // Login endpoint  
        $response = $this->post('/api/auth/login');
        $this->assertTrue(in_array($response->status(), [422, 401])); // Should not be 404

        // Verify phone endpoint
        $response = $this->post('/api/auth/verify-phone');
        $this->assertTrue(in_array($response->status(), [422, 401])); // Should not be 404
    }

    /**
     * Test protected API endpoints require authentication
     */
    public function test_protected_api_endpoints_require_authentication(): void
    {
        // Wallet endpoints
        $response = $this->get('/api/wallet');
        $this->assertEquals(401, $response->status());

        // Transaction endpoints
        $response = $this->get('/api/transactions');
        $this->assertEquals(401, $response->status());

        // KYC endpoints
        $response = $this->get('/api/kyc/status');
        $this->assertEquals(401, $response->status());
    }

    /**
     * Test webhook endpoints are accessible
     */
    public function test_webhook_endpoints_are_accessible(): void
    {
        // Paystack webhook
        $response = $this->post('/api/webhooks/paystack');
        $this->assertTrue(in_array($response->status(), [400, 422])); // Should not be 404

        // Monnify webhook
        $response = $this->post('/api/webhooks/monnify');
        $this->assertTrue(in_array($response->status(), [400, 422])); // Should not be 404
    }

    /**
     * Test models can be instantiated without errors
     */
    public function test_models_can_be_instantiated(): void
    {
        // Test User model
        $user = new User();
        $this->assertInstanceOf(User::class, $user);

        // Test Wallet model
        $wallet = new Wallet();
        $this->assertInstanceOf(Wallet::class, $wallet);

        // Test other critical models exist
        $this->assertTrue(class_exists(\App\Models\Transaction::class));
        $this->assertTrue(class_exists(\App\Models\VirtualAccount::class));
        $this->assertTrue(class_exists(\App\Models\VirtualCard::class));
        $this->assertTrue(class_exists(\App\Models\KycVerification::class));
    }

    /**
     * Test services can be instantiated
     */
    public function test_services_can_be_instantiated(): void
    {
        $this->assertTrue(class_exists(\App\Services\SmsService::class));
        $this->assertTrue(class_exists(\App\Services\PaystackService::class));
        $this->assertTrue(class_exists(\App\Services\MonnifyService::class));
        $this->assertTrue(class_exists(\App\Services\VTPassService::class));
    }
}