# BillingCS API Documentation

The BillingCS API provides RESTful endpoints for programmatic access to services, invoices, and tickets.

## Authentication

All API endpoints require authentication using Laravel Sanctum. Include the authentication token in your requests:

```
Authorization: Bearer {your-api-token}
```

## Endpoints

### User

#### Get Current User
```
GET /api/user
```

Returns the authenticated user's information.

**Response:**
```json
{
  "id": 1,
  "name": "John Doe",
  "email": "john@example.com",
  "role": "client"
}
```

### Services

#### List Services
```
GET /api/services
```

Returns paginated list of user's services.

**Parameters:**
- `page` (optional): Page number (default: 1)
- `per_page` (optional): Items per page (default: 15)

**Response:**
```json
{
  "data": [
    {
      "id": 1,
      "name": "Shared Hosting - Basic",
      "type": "hosting",
      "price": "9.99",
      "billing_cycle": "monthly",
      "status": "active",
      "next_due_date": "2025-02-01"
    }
  ],
  "meta": {
    "current_page": 1,
    "total": 10
  }
}
```

#### Get Service Details
```
GET /api/services/{id}
```

Returns detailed information about a specific service.

### Invoices

#### List Invoices
```
GET /api/invoices
```

Returns paginated list of user's invoices.

**Response:**
```json
{
  "data": [
    {
      "id": 1,
      "invoice_number": "INV-001",
      "date": "2025-01-01",
      "due_date": "2025-01-31",
      "total": "9.99",
      "status": "unpaid"
    }
  ]
}
```

#### Get Invoice Details
```
GET /api/invoices/{id}
```

Returns detailed information about a specific invoice including items.

**Response:**
```json
{
  "id": 1,
  "invoice_number": "INV-001",
  "date": "2025-01-01",
  "due_date": "2025-01-31",
  "subtotal": "9.99",
  "tax": "0.00",
  "total": "9.99",
  "status": "unpaid",
  "items": [
    {
      "id": 1,
      "description": "Shared Hosting - Basic - Monthly",
      "amount": "9.99",
      "quantity": 1
    }
  ]
}
```

### Tickets

#### List Tickets
```
GET /api/tickets
```

Returns paginated list of user's support tickets.

**Response:**
```json
{
  "data": [
    {
      "id": 1,
      "subject": "Server not responding",
      "status": "open",
      "priority": "high",
      "created_at": "2025-01-01T10:00:00Z"
    }
  ]
}
```

#### Get Ticket Details
```
GET /api/tickets/{id}
```

Returns detailed information about a specific ticket including replies.

**Response:**
```json
{
  "id": 1,
  "subject": "Server not responding",
  "status": "open",
  "priority": "high",
  "created_at": "2025-01-01T10:00:00Z",
  "replies": [
    {
      "id": 1,
      "message": "I can't access my server",
      "is_staff": false,
      "created_at": "2025-01-01T10:00:00Z"
    }
  ]
}
```

## Error Responses

### 401 Unauthorized
```json
{
  "message": "Unauthenticated."
}
```

### 403 Forbidden
```json
{
  "message": "This action is unauthorized."
}
```

### 404 Not Found
```json
{
  "message": "Resource not found."
}
```

### 422 Validation Error
```json
{
  "message": "The given data was invalid.",
  "errors": {
    "field_name": [
      "Error message"
    ]
  }
}
```

## Rate Limiting

API requests are rate-limited to prevent abuse:
- 60 requests per minute for authenticated users
- 10 requests per minute for unauthenticated requests

## Pagination

List endpoints return paginated results with the following structure:

```json
{
  "data": [...],
  "links": {
    "first": "http://example.com/api/resource?page=1",
    "last": "http://example.com/api/resource?page=10",
    "prev": null,
    "next": "http://example.com/api/resource?page=2"
  },
  "meta": {
    "current_page": 1,
    "from": 1,
    "last_page": 10,
    "per_page": 15,
    "to": 15,
    "total": 150
  }
}
```

## Example Usage

### cURL

```bash
curl -X GET "http://your-domain.com/api/services" \
  -H "Authorization: Bearer your-api-token" \
  -H "Accept: application/json"
```

### PHP (Guzzle)

```php
use GuzzleHttp\Client;

$client = new Client([
    'base_uri' => 'http://your-domain.com/api/',
    'headers' => [
        'Authorization' => 'Bearer ' . $token,
        'Accept' => 'application/json',
    ]
]);

$response = $client->get('services');
$data = json_decode($response->getBody(), true);
```

### JavaScript (Fetch)

```javascript
fetch('http://your-domain.com/api/services', {
  headers: {
    'Authorization': 'Bearer ' + token,
    'Accept': 'application/json'
  }
})
.then(response => response.json())
.then(data => console.log(data));
```

## Future Endpoints (Planned)

- POST /api/services - Create new service
- PUT /api/services/{id} - Update service
- POST /api/tickets - Create new ticket
- POST /api/tickets/{id}/reply - Reply to ticket
- POST /api/invoices/{id}/pay - Process payment

## Support

For API support and questions, please open an issue on GitHub.
