# BillingCS Installation Guide

This guide will walk you through the complete installation process of BillingCS, a Laravel 11 WHMCS-style hosting billing platform.

## System Requirements

### Minimum Requirements
- **PHP**: 8.2 or higher
- **Composer**: 2.0 or higher
- **Node.js**: 18.x or higher
- **NPM**: 9.x or higher
- **Database**: MySQL 8.0+ / MariaDB 10.6+ (recommended for production) or PostgreSQL 13+

### PHP Extensions Required
- OpenSSL PHP Extension
- PDO PHP Extension (with pdo_mysql for MySQL)
- Mbstring PHP Extension
- Tokenizer PHP Extension
- XML PHP Extension
- Ctype PHP Extension
- JSON PHP Extension
- BCMath PHP Extension

### Recommended
- Redis (for caching and queues)
- Supervisor (for queue workers in production)

## Installation Steps

### 1. Clone the Repository

```bash
git clone https://github.com/codemol3/BillingCS.git
cd BillingCS
```

### 2. Install Composer Dependencies

```bash
composer install
```

**Note**: If you encounter memory issues, try:
```bash
php -d memory_limit=-1 /usr/local/bin/composer install
```

### 3. Install NPM Dependencies

```bash
npm install
```

### 4. Environment Setup

Copy the example environment file and configure it:

```bash
cp .env.example .env
```

Generate application key:

```bash
php artisan key:generate
```

### 5. Database Configuration

#### Using MySQL (Recommended for Production)

The system is pre-configured to use MySQL. Edit your `.env` file:

```env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=billingcs
DB_USERNAME=root
DB_PASSWORD=your_password
```

Create the MySQL database:

```bash
mysql -u root -p
CREATE DATABASE billingcs CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
EXIT;
```

#### Alternative: Using PostgreSQL

If you prefer PostgreSQL, edit `.env` file:

```env
DB_CONNECTION=pgsql
DB_HOST=127.0.0.1
DB_PORT=5432
DB_DATABASE=billingcs
DB_USERNAME=postgres
DB_PASSWORD=your_password
```

Create the PostgreSQL database:

```bash
psql -U postgres
CREATE DATABASE billingcs;
\q
```

**Note**: For development purposes, you can still use SQLite by changing `DB_CONNECTION=sqlite` and creating `database/database.sqlite`, but MySQL is strongly recommended for production use.

### 6. Run Database Migrations

```bash
php artisan migrate
```

### 7. Seed the Database

This will create default admin and client users:

```bash
php artisan db:seed
```

**Default Credentials:**
- Admin: `admin@billingcs.com` / `password`
- Client: `client@billingcs.com` / `password`

### 8. Build Frontend Assets

For development:
```bash
npm run dev
```

For production:
```bash
npm run build
```

### 9. Set Permissions (Linux/Mac)

```bash
chmod -R 775 storage bootstrap/cache
chown -R www-data:www-data storage bootstrap/cache
```

Or for your current user:
```bash
chmod -R 775 storage bootstrap/cache
chown -R $USER:$USER storage bootstrap/cache
```

### 10. Start Development Server

```bash
php artisan serve
```

Visit: `http://localhost:8000`

## Additional Configuration

### Storage Link (Optional)

Create a symbolic link for public file storage:

```bash
php artisan storage:link
```

### Payment Gateway Configuration

#### PayPal Setup

Edit `.env`:

```env
PAYPAL_MODE=sandbox
PAYPAL_CLIENT_ID=your_paypal_client_id
PAYPAL_SECRET=your_paypal_secret
```

For production, change mode to `live` and use live credentials.

### Mail Configuration

#### Using Mailtrap (Development)

```env
MAIL_MAILER=smtp
MAIL_HOST=smtp.mailtrap.io
MAIL_PORT=2525
MAIL_USERNAME=your_username
MAIL_PASSWORD=your_password
MAIL_ENCRYPTION=tls
```

#### Using Gmail (Production - Less Secure Apps must be enabled)

```env
MAIL_MAILER=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_USERNAME=your_email@gmail.com
MAIL_PASSWORD=your_app_password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=your_email@gmail.com
MAIL_FROM_NAME="${APP_NAME}"
```

### Queue Configuration

For production, use database queues:

```env
QUEUE_CONNECTION=database
```

Run queue worker:
```bash
php artisan queue:work
```

### Caching Configuration

For better performance in production:

```env
CACHE_STORE=redis
SESSION_DRIVER=redis
```

Clear cache:
```bash
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

## Production Deployment

### 1. Web Server Configuration

#### Apache

Ensure `.htaccess` is working and `mod_rewrite` is enabled:

```bash
sudo a2enmod rewrite
sudo systemctl restart apache2
```

Document root should point to `public/` directory.

#### Nginx

Add this to your nginx config:

```nginx
server {
    listen 80;
    server_name your-domain.com;
    root /path/to/billingcs/public;

    add_header X-Frame-Options "SAMEORIGIN";
    add_header X-Content-Type-Options "nosniff";

    index index.php;

    charset utf-8;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location = /favicon.ico { access_log off; log_not_found off; }
    location = /robots.txt  { access_log off; log_not_found off; }

    error_page 404 /index.php;

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }

    location ~ /\.(?!well-known).* {
        deny all;
    }
}
```

### 2. Optimization

```bash
# Cache configuration
php artisan config:cache

# Cache routes
php artisan route:cache

# Cache views
php artisan view:cache

# Optimize autoloader
composer install --optimize-autoloader --no-dev
```

### 3. Set Production Environment

Edit `.env`:

```env
APP_ENV=production
APP_DEBUG=false
```

### 4. Setup Cron for Scheduled Tasks

Add to crontab:

```bash
* * * * * cd /path-to-project && php artisan schedule:run >> /dev/null 2>&1
```

### 5. Setup Supervisor for Queue Workers

Create `/etc/supervisor/conf.d/billingcs-worker.conf`:

```ini
[program:billingcs-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /path-to-project/artisan queue:work --sleep=3 --tries=3 --max-time=3600
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/path-to-project/storage/logs/worker.log
stopwaitsecs=3600
```

Reload supervisor:
```bash
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start billingcs-worker:*
```

## Troubleshooting

### Common Issues

#### "Class not found" errors
```bash
composer dump-autoload
```

#### Permission denied errors
```bash
sudo chmod -R 775 storage bootstrap/cache
sudo chown -R www-data:www-data storage bootstrap/cache
```

#### 500 Internal Server Error
Check Laravel logs:
```bash
tail -f storage/logs/laravel.log
```

#### Assets not loading
```bash
npm run build
php artisan storage:link
```

#### Database connection issues
- Verify database credentials in `.env`
- Check if database service is running
- Test connection: `php artisan db:show`

## Updating

To update to the latest version:

```bash
git pull origin main
composer install
npm install
php artisan migrate
php artisan cache:clear
npm run build
```

## Security Checklist

- [ ] Change default user passwords
- [ ] Set `APP_DEBUG=false` in production
- [ ] Set strong `APP_KEY`
- [ ] Enable HTTPS/SSL
- [ ] Configure proper file permissions
- [ ] Setup regular database backups
- [ ] Keep Laravel and dependencies updated
- [ ] Configure firewall rules
- [ ] Use environment-specific `.env` files
- [ ] Never commit `.env` to version control

## Getting Help

- **Documentation**: Check the README.md file
- **Issues**: https://github.com/codemol3/BillingCS/issues
- **Laravel Docs**: https://laravel.com/docs/11.x

## License

BillingCS is open-source software licensed under the MIT license.
