<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Service;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use Carbon\Carbon;

class GenerateRecurringInvoices extends Command
{
    protected $signature = 'billing:generate-invoices';
    protected $description = 'Generate recurring invoices for active services';

    public function handle()
    {
        $this->info('Starting invoice generation...');

        $services = Service::where('status', 'active')
            ->where('next_due_date', '<=', Carbon::now()->addDays(7))
            ->get();

        $count = 0;

        foreach ($services as $service) {
            $invoice = Invoice::create([
                'user_id' => $service->user_id,
                'invoice_number' => 'INV-' . strtoupper(uniqid()),
                'date' => Carbon::now(),
                'due_date' => $service->next_due_date,
                'subtotal' => $service->price,
                'tax' => 0,
                'total' => $service->price,
                'status' => 'unpaid',
            ]);

            InvoiceItem::create([
                'invoice_id' => $invoice->id,
                'description' => $service->name . ' - ' . ucfirst($service->billing_cycle),
                'amount' => $service->price,
                'quantity' => 1,
            ]);

            // Update next due date
            $service->update([
                'next_due_date' => $this->calculateNextDueDate($service->next_due_date, $service->billing_cycle),
            ]);

            $count++;
        }

        $this->info("Generated {$count} invoices successfully.");
        return Command::SUCCESS;
    }

    protected function calculateNextDueDate($currentDate, $cycle)
    {
        $date = Carbon::parse($currentDate);

        return match ($cycle) {
            'monthly' => $date->addMonth(),
            'quarterly' => $date->addMonths(3),
            'semi-annually' => $date->addMonths(6),
            'annually' => $date->addYear(),
            default => $date->addMonth(),
        };
    }
}
