<?php

namespace App\Http\Controllers\Admin\Billing;

use App\Http\Controllers\Controller;
use App\Models\Service;
use App\Models\Invoice;
use Illuminate\Http\Request;

class BillableItemController extends Controller
{
    public function uninvoiced()
    {
        // Get services that haven't been invoiced yet
        $items = Service::where('status', 'active')
                       ->whereDoesntHave('invoices', function($q) {
                           $q->where('status', 'unpaid');
                       })
                       ->with(['user', 'product'])
                       ->latest()
                       ->paginate(20);
        
        $stats = [
            'total_items' => $items->total(),
            'total_value' => $items->sum('price'),
        ];
        
        return view('admin.billing.billable.uninvoiced', compact('items', 'stats'));
    }

    public function recurring()
    {
        // Get all recurring services
        $items = Service::where('status', 'active')
                       ->whereNotNull('next_due_date')
                       ->with(['user', 'product'])
                       ->orderBy('next_due_date')
                       ->paginate(20);
        
        $stats = [
            'total_recurring' => $items->total(),
            'monthly_recurring' => Service::where('status', 'active')
                                         ->where('billing_cycle', 'monthly')
                                         ->count(),
            'annual_recurring' => Service::where('status', 'active')
                                        ->where('billing_cycle', 'annually')
                                        ->count(),
            'monthly_revenue' => Service::where('status', 'active')
                                       ->where('billing_cycle', 'monthly')
                                       ->sum('price'),
        ];
        
        return view('admin.billing.billable.recurring', compact('items', 'stats'));
    }

    public function create()
    {
        $clients = \App\Models\User::where('role', 'client')->get();
        $services = Service::where('status', 'active')->with('user')->get();
        
        return view('admin.billing.billable.create', compact('clients', 'services'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'description' => 'required|string',
            'amount' => 'required|numeric|min:0',
            'due_date' => 'required|date',
        ]);

        // Create an invoice for the billable item
        $user = \App\Models\User::findOrFail($validated['user_id']);
        
        $lastInvoice = Invoice::latest('id')->first();
        $nextNumber = $lastInvoice ? (int)substr($lastInvoice->invoice_number, 4) + 1 : 1;
        $invoiceNumber = 'INV-' . str_pad($nextNumber, 6, '0', STR_PAD_LEFT);

        $invoice = Invoice::create([
            'user_id' => $validated['user_id'],
            'invoice_number' => $invoiceNumber,
            'date' => now(),
            'due_date' => $validated['due_date'],
            'subtotal' => $validated['amount'],
            'tax' => 0,
            'total' => $validated['amount'],
            'status' => 'unpaid',
        ]);

        // Create invoice item
        \App\Models\InvoiceItem::create([
            'invoice_id' => $invoice->id,
            'description' => $validated['description'],
            'quantity' => 1,
            'unit_price' => $validated['amount'],
            'total' => $validated['amount'],
        ]);

        return redirect()->route('admin.billing.billable.uninvoiced')
                        ->with('success', 'Billable item created and invoice generated');
    }
}
