<?php

namespace App\Http\Controllers\Admin\Billing;

use App\Http\Controllers\Controller;
use App\Models\Quote;
use App\Models\User;
use App\Models\Product;
use Illuminate\Http\Request;

class QuoteController extends Controller
{
    public function index(Request $request)
    {
        $query = Quote::with('user');

        if ($request->has('status')) {
            $query->where('status', $request->get('status'));
        }

        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function($q) use ($search) {
                $q->where('quote_number', 'like', "%{$search}%")
                  ->orWhere('subject', 'like', "%{$search}%")
                  ->orWhereHas('user', function($subQ) use ($search) {
                      $subQ->where('name', 'like', "%{$search}%")
                           ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        $quotes = $query->latest()->paginate(20);
        
        $stats = [
            'total_quotes' => Quote::count(),
            'pending_quotes' => Quote::where('status', 'pending')->count(),
            'accepted_quotes' => Quote::where('status', 'accepted')->count(),
            'expired_quotes' => Quote::where('status', 'expired')->count(),
            'total_value' => Quote::where('status', 'accepted')->sum('total'),
        ];
        
        return view('admin.billing.quotes.index', compact('quotes', 'stats'));
    }

    public function show($id)
    {
        $quote = Quote::with('user')->findOrFail($id);
        
        return view('admin.billing.quotes.show', compact('quote'));
    }

    public function create()
    {
        $clients = User::where('role', 'client')->get();
        $products = Product::where('active', true)->get();
        
        return view('admin.billing.quotes.create', compact('clients', 'products'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'subject' => 'required|string|max:255',
            'date' => 'required|date',
            'expiry_date' => 'required|date|after:date',
            'subtotal' => 'required|numeric|min:0',
            'tax' => 'nullable|numeric|min:0',
            'total' => 'required|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        // Generate quote number
        $lastQuote = Quote::latest('id')->first();
        $nextNumber = $lastQuote ? (int)substr($lastQuote->quote_number, 3) + 1 : 1;
        $validated['quote_number'] = 'QT-' . str_pad($nextNumber, 6, '0', STR_PAD_LEFT);
        $validated['status'] = 'pending';

        $quote = Quote::create($validated);

        return redirect()->route('admin.billing.quotes')
                        ->with('success', 'Quote created successfully');
    }
}
