<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\User;
use App\Models\Service;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    public function index()
    {
        return view('admin.reports.index');
    }

    public function revenue(Request $request)
    {
        $startDate = $request->input('start_date', now()->subDays(30));
        $endDate = $request->input('end_date', now());

        // Revenue by day
        $revenueData = Invoice::where('status', 'paid')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('SUM(total) as total'))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Revenue by gateway
        $revenueByGateway = Transaction::where('status', 'completed')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->select('gateway', DB::raw('SUM(amount) as total'))
            ->groupBy('gateway')
            ->get();

        // Total metrics
        $totalRevenue = Invoice::where('status', 'paid')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->sum('total');

        $totalInvoices = Invoice::whereBetween('created_at', [$startDate, $endDate])->count();
        $paidInvoices = Invoice::where('status', 'paid')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->count();

        return view('admin.reports.revenue', compact(
            'revenueData',
            'revenueByGateway',
            'totalRevenue',
            'totalInvoices',
            'paidInvoices',
            'startDate',
            'endDate'
        ));
    }

    public function users(Request $request)
    {
        $startDate = $request->input('start_date', now()->subDays(30));
        $endDate = $request->input('end_date', now());

        // New users by day
        $userData = User::whereBetween('created_at', [$startDate, $endDate])
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as count'))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // User stats
        $totalUsers = User::whereBetween('created_at', [$startDate, $endDate])->count();
        $clientUsers = User::where('role', 'client')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->count();

        return view('admin.reports.users', compact(
            'userData',
            'totalUsers',
            'clientUsers',
            'startDate',
            'endDate'
        ));
    }

    public function services(Request $request)
    {
        $startDate = $request->input('start_date', now()->subDays(30));
        $endDate = $request->input('end_date', now());

        // Services by status
        $servicesByStatus = Service::whereBetween('created_at', [$startDate, $endDate])
            ->select('status', DB::raw('COUNT(*) as count'))
            ->groupBy('status')
            ->get();

        // Services by type
        $servicesByType = Service::whereBetween('created_at', [$startDate, $endDate])
            ->select('type', DB::raw('COUNT(*) as count'))
            ->groupBy('type')
            ->get();

        $totalServices = Service::whereBetween('created_at', [$startDate, $endDate])->count();
        $activeServices = Service::where('status', 'active')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->count();

        return view('admin.reports.services', compact(
            'servicesByStatus',
            'servicesByType',
            'totalServices',
            'activeServices',
            'startDate',
            'endDate'
        ));
    }

    public function export(Request $request)
    {
        $type = $request->input('type'); // users, invoices, services, transactions
        $format = $request->input('format', 'csv'); // csv or pdf

        $data = $this->getExportData($type);
        
        if ($format === 'csv') {
            return $this->exportToCsv($data, $type);
        } else {
            return $this->exportToPdf($data, $type);
        }
    }

    private function getExportData($type)
    {
        switch ($type) {
            case 'users':
                return User::with('services', 'invoices')->get();
            case 'invoices':
                return Invoice::with('user', 'items')->get();
            case 'services':
                return Service::with('user')->get();
            case 'transactions':
                return Transaction::with('invoice')->get();
            default:
                return collect();
        }
    }

    private function exportToCsv($data, $type)
    {
        $filename = $type . '_export_' . now()->format('Y-m-d') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($data, $type) {
            $file = fopen('php://output', 'w');
            
            // Add headers based on type
            $headers = $this->getCsvHeaders($type);
            fputcsv($file, $headers);

            // Add data rows
            foreach ($data as $row) {
                fputcsv($file, $this->formatCsvRow($row, $type));
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function getCsvHeaders($type)
    {
        switch ($type) {
            case 'users':
                return ['ID', 'Name', 'Email', 'Role', 'Created At'];
            case 'invoices':
                return ['ID', 'User', 'Total', 'Status', 'Due Date', 'Created At'];
            case 'services':
                return ['ID', 'User', 'Name', 'Type', 'Status', 'Price', 'Created At'];
            case 'transactions':
                return ['ID', 'Invoice', 'Gateway', 'Amount', 'Status', 'Reference', 'Created At'];
            default:
                return [];
        }
    }

    private function formatCsvRow($row, $type)
    {
        switch ($type) {
            case 'users':
                return [$row->id, $row->name, $row->email, $row->role, $row->created_at];
            case 'invoices':
                return [$row->id, $row->user->name ?? 'N/A', $row->total, $row->status, $row->due_date, $row->created_at];
            case 'services':
                return [$row->id, $row->user->name ?? 'N/A', $row->name, $row->type, $row->status, $row->price, $row->created_at];
            case 'transactions':
                return [$row->id, $row->invoice_id, $row->gateway, $row->amount, $row->status, $row->reference, $row->created_at];
            default:
                return [];
        }
    }

    private function exportToPdf($data, $type)
    {
        // Generate simple PDF export using Laravel's response
        $filename = $type . '_export_' . now()->format('Y-m-d') . '.pdf';
        
        $html = view('admin.reports.pdf-export', [
            'data' => $data,
            'type' => $type,
            'date' => now()->format('Y-m-d H:i:s')
        ])->render();
        
        // For production, install and use a PDF library like DomPDF or TCPDF
        // For now, return HTML response that can be printed to PDF
        return response($html)->header('Content-Type', 'text/html');
    }
}
