<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use ZipArchive;

class TemplateManagerController extends Controller
{
    public function index()
    {
        $templates = \App\Models\Template::orderBy('name')->get();
        return view('admin.templates.index', compact('templates'));
    }

    public function upload(Request $request)
    {
        $request->validate([
            'template_file' => 'required|file|mimes:zip|max:20480',
        ]);

        $file = $request->file('template_file');
        $zip = new ZipArchive;
        
        if ($zip->open($file->getPathname()) === TRUE) {
            // Extract to temporary directory
            $tempPath = storage_path('app/temp/' . uniqid());
            $zip->extractTo($tempPath);
            $zip->close();

            // Validate template.json
            $templateJsonPath = $tempPath . '/template.json';
            if (!file_exists($templateJsonPath)) {
                File::deleteDirectory($tempPath);
                return back()->with('error', 'Invalid template: template.json not found');
            }

            $templateConfig = json_decode(file_get_contents($templateJsonPath), true);
            if (!$templateConfig || !isset($templateConfig['name'])) {
                File::deleteDirectory($tempPath);
                return back()->with('error', 'Invalid template.json format');
            }

            // Determine template destination
            $destinationPath = resource_path('views/templates/' . \Illuminate\Support\Str::slug($templateConfig['name']));
            
            // Check if template already exists
            if (File::exists($destinationPath)) {
                File::deleteDirectory($tempPath);
                return back()->with('error', 'Template already exists');
            }

            // Move template to final location
            File::moveDirectory($tempPath, $destinationPath);

            // Register template in database
            \App\Models\Template::create([
                'name' => $templateConfig['name'],
                'slug' => \Illuminate\Support\Str::slug($templateConfig['name']),
                'version' => $templateConfig['version'] ?? '1.0.0',
                'author' => $templateConfig['author'] ?? 'Unknown',
                'active' => false,
                'config' => json_encode($templateConfig),
            ]);

            return redirect()->route('admin.templates.index')
                ->with('success', 'Template uploaded and installed successfully');
        }

        return back()->with('error', 'Failed to extract template ZIP file');
    }

    public function activate($id)
    {
        // Deactivate all templates
        \App\Models\Template::query()->update(['active' => false]);
        
        // Activate selected template
        $template = \App\Models\Template::findOrFail($id);
        $template->active = true;
        $template->save();

        // Update setting
        \App\Models\Setting::updateOrCreate(
            ['key' => 'active_template'],
            ['value' => $template->slug]
        );

        return back()->with('success', 'Template activated successfully');
    }

    public function uninstall($id)
    {
        $template = \App\Models\Template::findOrFail($id);
        
        if ($template->active) {
            return back()->with('error', 'Cannot uninstall active template');
        }
        
        // Delete template files
        $templatePath = resource_path('views/templates/' . $template->slug);
        
        if (File::exists($templatePath)) {
            File::deleteDirectory($templatePath);
        }

        // Delete from database
        $template->delete();

        return redirect()->route('admin.templates.index')
            ->with('success', 'Template uninstalled successfully');
    }
}
