<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\DomainPricing;
use Illuminate\Http\Request;

class CartController extends Controller
{
    public function index()
    {
        $cart = session()->get('cart', []);
        $total = $this->calculateTotal($cart);
        
        return view('client.cart.index', compact('cart', 'total'));
    }

    public function addProduct(Request $request, Product $product)
    {
        $validated = $request->validate([
            'billing_cycle' => 'required|string',
            'domain' => 'nullable|string',
        ]);

        $pricing = $product->pricing()->where('billing_cycle', $validated['billing_cycle'])->first();
        
        if (!$pricing) {
            return back()->with('error', 'Invalid billing cycle selected');
        }

        $cart = session()->get('cart', []);
        
        $cartItem = [
            'type' => 'product',
            'product_id' => $product->id,
            'name' => $product->name,
            'billing_cycle' => $validated['billing_cycle'],
            'price' => $pricing->price,
            'setup_fee' => $pricing->setup_fee,
            'domain' => $validated['domain'] ?? null,
        ];

        $cart[] = $cartItem;
        session()->put('cart', $cart);

        return redirect()->route('cart.index')->with('success', 'Product added to cart');
    }

    public function addDomain(Request $request)
    {
        $validated = $request->validate([
            'domain' => 'required|string',
            'extension' => 'required|string',
            'action' => 'required|in:register,transfer',
        ]);

        $domainPricing = DomainPricing::where('extension', $validated['extension'])->first();
        
        if (!$domainPricing) {
            return back()->with('error', 'Domain extension not found');
        }

        $price = $validated['action'] === 'register' ? $domainPricing->register_price : $domainPricing->transfer_price;

        $cart = session()->get('cart', []);
        
        $cartItem = [
            'type' => 'domain',
            'domain' => $validated['domain'] . $validated['extension'],
            'action' => $validated['action'],
            'price' => $price,
            'setup_fee' => 0,
        ];

        $cart[] = $cartItem;
        session()->put('cart', $cart);

        return redirect()->route('cart.index')->with('success', 'Domain added to cart');
    }

    public function remove(Request $request, $index)
    {
        $cart = session()->get('cart', []);
        
        if (isset($cart[$index])) {
            unset($cart[$index]);
            $cart = array_values($cart);
            session()->put('cart', $cart);
        }

        return redirect()->route('cart.index')->with('success', 'Item removed from cart');
    }

    public function clear()
    {
        session()->forget('cart');
        return redirect()->route('cart.index')->with('success', 'Cart cleared');
    }

    public function checkout(Request $request)
    {
        $cart = session()->get('cart', []);
        
        if (empty($cart)) {
            return redirect()->route('cart.index')->with('error', 'Cart is empty');
        }

        $validated = $request->validate([
            'payment_gateway' => 'required|string',
        ]);

        // Create invoice and invoice items
        $user = auth()->user();
        $total = $this->calculateTotal($cart);

        $invoice = $user->invoices()->create([
            'total' => $total,
            'status' => 'unpaid',
            'due_date' => now()->addDays(7),
            'currency' => setting('currency', 'USD'),
        ]);

        foreach ($cart as $item) {
            $invoice->items()->create([
                'description' => $item['type'] === 'domain' 
                    ? ucfirst($item['action']) . ' Domain: ' . $item['domain']
                    : $item['name'] . ' (' . $item['billing_cycle'] . ')',
                'quantity' => 1,
                'unit_price' => $item['price'],
                'total' => $item['price'] + $item['setup_fee'],
            ]);

            // Create service for hosting products
            if ($item['type'] === 'product') {
                $user->services()->create([
                    'name' => $item['name'],
                    'type' => 'hosting',
                    'billing_cycle' => $item['billing_cycle'],
                    'price' => $item['price'],
                    'status' => 'pending',
                    'domain' => $item['domain'] ?? null,
                ]);
            }
        }

        // Clear cart
        session()->forget('cart');

        return redirect()->route('client.invoices.show', $invoice)
            ->with('success', 'Order placed successfully. Please complete payment.');
    }

    private function calculateTotal($cart)
    {
        $total = 0;
        foreach ($cart as $item) {
            $total += $item['price'] + $item['setup_fee'];
        }
        return $total;
    }
}
