<?php

namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class AccountController extends Controller
{
    public function overview()
    {
        $user = auth()->user();
        
        $stats = [
            'total_services' => $user->services()->count(),
            'active_services' => $user->services()->where('status', 'active')->count(),
            'total_invoices' => $user->invoices()->count(),
            'unpaid_invoices' => $user->invoices()->where('status', 'unpaid')->count(),
            'open_tickets' => $user->tickets()->where('status', 'open')->count(),
            'total_spent' => $user->invoices()->where('status', 'paid')->sum('total'),
            'account_balance' => $user->transactions()->where('status', 'completed')->sum('amount'),
        ];
        
        $recentActivity = [
            'services' => $user->services()->latest()->limit(5)->get(),
            'invoices' => $user->invoices()->latest()->limit(5)->get(),
            'tickets' => $user->tickets()->latest()->limit(5)->get(),
        ];
        
        return view('client.account.overview', compact('stats', 'recentActivity'));
    }

    public function profile()
    {
        $user = auth()->user();
        
        return view('client.account.profile', compact('user'));
    }

    public function updateProfile(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . auth()->id(),
            'company' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'zip' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',
        ]);

        auth()->user()->update($validated);

        return back()->with('success', 'Profile updated successfully');
    }

    public function security()
    {
        return view('client.account.security');
    }

    public function updatePassword(Request $request)
    {
        $validated = $request->validate([
            'current_password' => 'required',
            'password' => 'required|min:8|confirmed',
        ]);

        if (!\Hash::check($validated['current_password'], auth()->user()->password)) {
            return back()->withErrors(['current_password' => 'Current password is incorrect']);
        }

        auth()->user()->update([
            'password' => bcrypt($validated['password']),
        ]);

        return back()->with('success', 'Password updated successfully');
    }

    public function contacts()
    {
        $user = auth()->user();
        
        return view('client.account.contacts', compact('user'));
    }

    public function addFunds()
    {
        return view('client.account.add-funds');
    }

    public function messages()
    {
        $messages = auth()->user()->messages()
                                  ->latest()
                                  ->paginate(20);
        
        return view('client.account.messages', compact('messages'));
    }
}
