<?php

namespace App\Http\Controllers;

use App\Models\DomainPricing;
use Illuminate\Http\Request;

class DomainController extends Controller
{
    public function check(Request $request)
    {
        $validated = $request->validate([
            'domain' => 'required|string|max:255',
        ]);

        $domainParts = $this->parseDomain($validated['domain']);
        
        if (!$domainParts) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid domain format',
            ]);
        }

        $extension = $domainParts['extension'];
        $domainPricing = DomainPricing::where('extension', $extension)->where('active', true)->first();

        if (!$domainPricing) {
            return response()->json([
                'success' => false,
                'message' => 'Domain extension not supported',
                'extension' => $extension,
            ]);
        }

        // Simulate domain availability check (in production, integrate with registrar API)
        $available = $this->checkAvailability($validated['domain']);

        return response()->json([
            'success' => true,
            'domain' => $validated['domain'],
            'available' => $available,
            'pricing' => [
                'register' => $domainPricing->register_price,
                'renew' => $domainPricing->renew_price,
                'transfer' => $domainPricing->transfer_price,
            ],
            'message' => $available ? 'Domain is available!' : 'Domain is already registered',
        ]);
    }

    private function parseDomain($domain)
    {
        // Remove http://, https://, and www.
        $domain = preg_replace('#^https?://(www\.)?#i', '', $domain);
        
        // Extract extension
        $parts = explode('.', $domain);
        
        if (count($parts) < 2) {
            return null;
        }

        $extension = '.' . end($parts);
        $name = implode('.', array_slice($parts, 0, -1));

        return [
            'name' => $name,
            'extension' => $extension,
            'full' => $domain,
        ];
    }

    private function checkAvailability($domain)
    {
        // Simulate availability check
        // In production, integrate with domain registrar API (e.g., ResellerClub, Namecheap, etc.)
        // For now, use a simple hash-based simulation
        $hash = crc32($domain);
        return ($hash % 3) !== 0; // ~66% chance of being available
    }
}
