<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\Transaction;
use App\Models\PaymentGateway;
use Illuminate\Http\Request;

class WebhookController extends Controller
{
    public function paypal(Request $request)
    {
        // Handle PayPal IPN/Webhook
        $gateway = PaymentGateway::where('slug', 'paypal')->first();
        
        if (!$gateway || !$gateway->is_active) {
            return response()->json(['error' => 'Gateway not active'], 400);
        }

        $paypalGateway = new \App\Modules\PaymentGateways\PayPal\PayPalGateway($gateway->config);
        
        $headers = [
            'PAYPAL-TRANSMISSION-ID' => $request->header('PAYPAL-TRANSMISSION-ID'),
            'PAYPAL-TRANSMISSION-TIME' => $request->header('PAYPAL-TRANSMISSION-TIME'),
            'PAYPAL-CERT-URL' => $request->header('PAYPAL-CERT-URL'),
            'PAYPAL-AUTH-ALGO' => $request->header('PAYPAL-AUTH-ALGO'),
            'PAYPAL-TRANSMISSION-SIG' => $request->header('PAYPAL-TRANSMISSION-SIG'),
        ];
        
        $result = $paypalGateway->handleWebhook($request->getContent(), $headers);
        
        if ($result['success']) {
            // Process the webhook event (e.g., mark invoice as paid)
            // Implementation depends on event type
            return response()->json(['status' => 'success']);
        }
        
        return response()->json(['error' => 'Invalid webhook'], 400);
    }

    public function stripe(Request $request)
    {
        // Handle Stripe Webhook
        $gateway = PaymentGateway::where('slug', 'stripe')->first();
        
        if (!$gateway || !$gateway->is_active) {
            return response()->json(['error' => 'Gateway not active'], 400);
        }

        $stripeGateway = new \App\Modules\PaymentGateways\Stripe\StripeGateway($gateway->config);
        
        $signature = $request->header('Stripe-Signature');
        $result = $stripeGateway->handleWebhook($request->getContent(), $signature);
        
        if ($result['success']) {
            $eventType = $result['type'];
            $data = $result['data'];
            
            if ($eventType === 'payment_intent.succeeded') {
                // Find and mark invoice as paid
                $invoiceId = $data['metadata']['invoice_id'] ?? null;
                if ($invoiceId) {
                    $invoice = Invoice::find($invoiceId);
                    if ($invoice) {
                        $invoice->update(['status' => 'paid']);
                        
                        Transaction::create([
                            'invoice_id' => $invoice->id,
                            'gateway' => 'stripe',
                            'transaction_id' => $data['id'],
                            'amount' => $data['amount'] / 100,
                            'status' => 'completed',
                        ]);
                    }
                }
            }
            
            return response()->json(['status' => 'success']);
        }
        
        return response()->json(['error' => 'Invalid webhook'], 400);
    }

    public function paystack(Request $request)
    {
        // Handle Paystack Webhook
        $gateway = PaymentGateway::where('slug', 'paystack')->first();
        
        if (!$gateway || !$gateway->is_active) {
            return response()->json(['error' => 'Gateway not active'], 400);
        }

        $paystackGateway = new \App\Modules\PaymentGateways\Paystack\PaystackGateway($gateway->config);
        
        $signature = $request->header('X-Paystack-Signature');
        $result = $paystackGateway->handleWebhook($request->getContent(), $signature);
        
        if ($result['success']) {
            $event = $result['event'];
            $data = $result['data'];
            
            if ($event === 'charge.success') {
                // Find and mark invoice as paid
                $invoiceId = $data['metadata']['invoice_id'] ?? null;
                if ($invoiceId) {
                    $invoice = Invoice::find($invoiceId);
                    if ($invoice) {
                        $invoice->update(['status' => 'paid']);
                        
                        Transaction::create([
                            'invoice_id' => $invoice->id,
                            'gateway' => 'paystack',
                            'transaction_id' => $data['reference'],
                            'amount' => $data['amount'] / 100,
                            'status' => 'completed',
                        ]);
                    }
                }
            }
            
            return response()->json(['status' => 'success']);
        }
        
        return response()->json(['error' => 'Invalid webhook'], 400);
    }

    public function flutterwave(Request $request)
    {
        // Handle Flutterwave Webhook
        $gateway = PaymentGateway::where('slug', 'flutterwave')->first();
        
        if (!$gateway || !$gateway->is_active) {
            return response()->json(['error' => 'Gateway not active'], 400);
        }

        $flutterwaveGateway = new \App\Modules\PaymentGateways\Flutterwave\FlutterwaveGateway($gateway->config);
        
        $signature = $request->header('verif-hash');
        $result = $flutterwaveGateway->handleWebhook($request->getContent(), $signature);
        
        if ($result['success']) {
            $event = $result['event'];
            $data = $result['data'];
            
            if ($event === 'charge.completed') {
                // Find and mark invoice as paid
                $invoiceId = $data['meta']['invoice_id'] ?? null;
                if ($invoiceId) {
                    $invoice = Invoice::find($invoiceId);
                    if ($invoice) {
                        $invoice->update(['status' => 'paid']);
                        
                        Transaction::create([
                            'invoice_id' => $invoice->id,
                            'gateway' => 'flutterwave',
                            'transaction_id' => $data['tx_ref'],
                            'amount' => $data['amount'],
                            'status' => 'completed',
                        ]);
                    }
                }
            }
            
            return response()->json(['status' => 'success']);
        }
        
        return response()->json(['error' => 'Invalid webhook'], 400);
    }

    public function razorpay(Request $request)
    {
        // Handle Razorpay Webhook
        $gateway = PaymentGateway::where('slug', 'razorpay')->first();
        
        if (!$gateway || !$gateway->is_active) {
            return response()->json(['error' => 'Gateway not active'], 400);
        }

        $razorpayGateway = new \App\Modules\PaymentGateways\Razorpay\RazorpayGateway($gateway->config);
        
        $signature = $request->header('X-Razorpay-Signature');
        $result = $razorpayGateway->handleWebhook($request->getContent(), $signature);
        
        if ($result['success']) {
            $event = $result['event'];
            $entity = $result['entity'];
            
            if ($event === 'payment.captured') {
                // Find and mark invoice as paid
                $invoiceId = $entity['notes']['invoice_id'] ?? null;
                if ($invoiceId) {
                    $invoice = Invoice::find($invoiceId);
                    if ($invoice) {
                        $invoice->update(['status' => 'paid']);
                        
                        Transaction::create([
                            'invoice_id' => $invoice->id,
                            'gateway' => 'razorpay',
                            'transaction_id' => $entity['id'],
                            'amount' => $entity['amount'] / 100,
                            'status' => 'completed',
                        ]);
                    }
                }
            }
            
            return response()->json(['status' => 'success']);
        }
        
        return response()->json(['error' => 'Invalid webhook'], 400);
    }
}
