<?php

namespace App\Modules\DomainRegistrars\InternetBS;

class InternetBSRegistrar
{
    protected $config;
    protected $apiKey;
    protected $password;
    protected $baseUrl = 'https://api.internet.bs';

    public function __construct($config = [])
    {
        $this->config = $config;
        $this->apiKey = $config['api_key'] ?? env('INTERNETBS_API_KEY');
        $this->password = $config['password'] ?? env('INTERNETBS_PASSWORD');
        
        $testMode = $config['test_mode'] ?? env('INTERNETBS_TEST_MODE', true);
        if ($testMode) {
            $this->baseUrl = 'https://testapi.internet.bs';
        }
    }

    public function checkAvailability($domain)
    {
        // Real Internet.bs API Domain Check
        // Documentation: https://internetbs.net/ResellerRegistrarDomainNameAPI/api/01_check_domain_availability
        
        $params = [
            'ApiKey' => $this->apiKey,
            'Password' => $this->password,
            'Domain' => $domain,
            'ResponseFormat' => 'json',
        ];

        $url = $this->baseUrl . '/Domain/Check?' . http_build_query($params);
        
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        $data = json_decode($response, true);
        
        if (isset($data['status']) && $data['status'] === 'SUCCESS') {
            return [
                'success' => true,
                'available' => strtolower($data['availability']) === 'available',
                'domain' => $domain,
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Domain check failed',
        ];
    }

    public function registerDomain($domain, $years, $contactInfo)
    {
        // Real Internet.bs API Domain Registration
        // Documentation: https://internetbs.net/ResellerRegistrarDomainNameAPI/api/02_domain_create
        
        $params = [
            'ApiKey' => $this->apiKey,
            'Password' => $this->password,
            'Domain' => $domain,
            'Period' => $years . 'Y',
            'Registrant_FirstName' => $contactInfo['first_name'],
            'Registrant_LastName' => $contactInfo['last_name'],
            'Registrant_Email' => $contactInfo['email'],
            'Registrant_PhoneNumber' => $contactInfo['phone'],
            'Registrant_Street' => $contactInfo['address'],
            'Registrant_City' => $contactInfo['city'],
            'Registrant_CountryCode' => $contactInfo['country'],
            'Registrant_PostalCode' => $contactInfo['zip'],
            'Technical_FirstName' => $contactInfo['first_name'],
            'Technical_LastName' => $contactInfo['last_name'],
            'Technical_Email' => $contactInfo['email'],
            'Technical_PhoneNumber' => $contactInfo['phone'],
            'Technical_Street' => $contactInfo['address'],
            'Technical_City' => $contactInfo['city'],
            'Technical_CountryCode' => $contactInfo['country'],
            'Technical_PostalCode' => $contactInfo['zip'],
            'Admin_FirstName' => $contactInfo['first_name'],
            'Admin_LastName' => $contactInfo['last_name'],
            'Admin_Email' => $contactInfo['email'],
            'Admin_PhoneNumber' => $contactInfo['phone'],
            'Admin_Street' => $contactInfo['address'],
            'Admin_City' => $contactInfo['city'],
            'Admin_CountryCode' => $contactInfo['country'],
            'Admin_PostalCode' => $contactInfo['zip'],
            'Billing_FirstName' => $contactInfo['first_name'],
            'Billing_LastName' => $contactInfo['last_name'],
            'Billing_Email' => $contactInfo['email'],
            'Billing_PhoneNumber' => $contactInfo['phone'],
            'Billing_Street' => $contactInfo['address'],
            'Billing_City' => $contactInfo['city'],
            'Billing_CountryCode' => $contactInfo['country'],
            'Billing_PostalCode' => $contactInfo['zip'],
            'ResponseFormat' => 'json',
        ];

        $url = $this->baseUrl . '/Domain/Create?' . http_build_query($params);
        
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        $data = json_decode($response, true);
        
        if (isset($data['status']) && $data['status'] === 'SUCCESS') {
            return [
                'success' => true,
                'domain' => $domain,
                'transaction_id' => $data['transactid'] ?? '',
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Domain registration failed',
        ];
    }

    public function renewDomain($domain, $years)
    {
        // Real Internet.bs API Domain Renewal
        // Documentation: https://internetbs.net/ResellerRegistrarDomainNameAPI/api/03_domain_renew
        
        $params = [
            'ApiKey' => $this->apiKey,
            'Password' => $this->password,
            'Domain' => $domain,
            'Period' => $years . 'Y',
            'ResponseFormat' => 'json',
        ];

        $url = $this->baseUrl . '/Domain/Renew?' . http_build_query($params);
        
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        $data = json_decode($response, true);
        
        if (isset($data['status']) && $data['status'] === 'SUCCESS') {
            return [
                'success' => true,
                'domain' => $domain,
                'transaction_id' => $data['transactid'] ?? '',
                'expiration_date' => $data['expirationdate'] ?? '',
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Domain renewal failed',
        ];
    }

    public function transferDomain($domain, $eppCode, $years)
    {
        // Real Internet.bs API Domain Transfer
        // Documentation: https://internetbs.net/ResellerRegistrarDomainNameAPI/api/04_domain_transfer_initiate
        
        $params = [
            'ApiKey' => $this->apiKey,
            'Password' => $this->password,
            'Domain' => $domain,
            'TransferAuthInfo' => $eppCode,
            'ResponseFormat' => 'json',
        ];

        $url = $this->baseUrl . '/Domain/Transfer/Initiate?' . http_build_query($params);
        
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        $data = json_decode($response, true);
        
        if (isset($data['status']) && $data['status'] === 'SUCCESS') {
            return [
                'success' => true,
                'domain' => $domain,
                'transaction_id' => $data['transactid'] ?? '',
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Domain transfer failed',
        ];
    }
}
