<?php

namespace App\Modules\DomainRegistrars\Whogohost;

class WhogohostRegistrar
{
    protected $config;
    protected $apiKey;
    protected $baseUrl = 'https://api.whogohost.com/v1';

    public function __construct($config = [])
    {
        $this->config = $config;
        $this->apiKey = $config['api_key'] ?? env('WHOGOHOST_API_KEY');
    }

    public function checkAvailability($domain)
    {
        // Whogohost/Nigerian Registrar API Domain Check
        // Supports .ng, .com.ng, .org.ng, .net.ng, .edu.ng domains
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->baseUrl . '/domains/check',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->apiKey,
                'Content-Type: application/json',
            ],
            CURLOPT_POSTFIELDS => json_encode([
                'domain' => $domain,
            ])
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);
        
        if ($httpCode === 200 && isset($data['status'])) {
            return [
                'success' => true,
                'available' => $data['available'] === true,
                'domain' => $domain,
                'price' => $data['price'] ?? null,
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Domain check failed',
        ];
    }

    public function registerDomain($domain, $years, $contactInfo)
    {
        // Whogohost/Nigerian Registrar API Domain Registration
        // Full support for Nigerian TLDs (.ng, .com.ng, .org.ng, etc.)
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->baseUrl . '/domains/register',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->apiKey,
                'Content-Type: application/json',
            ],
            CURLOPT_POSTFIELDS => json_encode([
                'domain' => $domain,
                'period' => $years,
                'registrant' => [
                    'first_name' => $contactInfo['first_name'],
                    'last_name' => $contactInfo['last_name'],
                    'email' => $contactInfo['email'],
                    'phone' => $contactInfo['phone'],
                    'address' => $contactInfo['address'],
                    'city' => $contactInfo['city'],
                    'state' => $contactInfo['state'],
                    'country' => $contactInfo['country'],
                    'postal_code' => $contactInfo['zip'],
                ],
                'admin' => [
                    'first_name' => $contactInfo['first_name'],
                    'last_name' => $contactInfo['last_name'],
                    'email' => $contactInfo['email'],
                    'phone' => $contactInfo['phone'],
                    'address' => $contactInfo['address'],
                    'city' => $contactInfo['city'],
                    'state' => $contactInfo['state'],
                    'country' => $contactInfo['country'],
                    'postal_code' => $contactInfo['zip'],
                ],
                'technical' => [
                    'first_name' => $contactInfo['first_name'],
                    'last_name' => $contactInfo['last_name'],
                    'email' => $contactInfo['email'],
                    'phone' => $contactInfo['phone'],
                    'address' => $contactInfo['address'],
                    'city' => $contactInfo['city'],
                    'state' => $contactInfo['state'],
                    'country' => $contactInfo['country'],
                    'postal_code' => $contactInfo['zip'],
                ],
                'billing' => [
                    'first_name' => $contactInfo['first_name'],
                    'last_name' => $contactInfo['last_name'],
                    'email' => $contactInfo['email'],
                    'phone' => $contactInfo['phone'],
                    'address' => $contactInfo['address'],
                    'city' => $contactInfo['city'],
                    'state' => $contactInfo['state'],
                    'country' => $contactInfo['country'],
                    'postal_code' => $contactInfo['zip'],
                ],
                'nameservers' => $contactInfo['nameservers'] ?? [
                    'ns1.whogohost.com',
                    'ns2.whogohost.com',
                ],
            ])
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);
        
        if ($httpCode === 200 && isset($data['status']) && $data['status'] === 'success') {
            return [
                'success' => true,
                'domain' => $domain,
                'transaction_id' => $data['transaction_id'] ?? '',
                'expiration_date' => $data['expiration_date'] ?? '',
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Domain registration failed',
        ];
    }

    public function renewDomain($domain, $years)
    {
        // Whogohost/Nigerian Registrar API Domain Renewal
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->baseUrl . '/domains/renew',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->apiKey,
                'Content-Type: application/json',
            ],
            CURLOPT_POSTFIELDS => json_encode([
                'domain' => $domain,
                'period' => $years,
            ])
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);
        
        if ($httpCode === 200 && isset($data['status']) && $data['status'] === 'success') {
            return [
                'success' => true,
                'domain' => $domain,
                'transaction_id' => $data['transaction_id'] ?? '',
                'expiration_date' => $data['expiration_date'] ?? '',
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Domain renewal failed',
        ];
    }

    public function transferDomain($domain, $eppCode, $years)
    {
        // Whogohost/Nigerian Registrar API Domain Transfer
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->baseUrl . '/domains/transfer',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->apiKey,
                'Content-Type: application/json',
            ],
            CURLOPT_POSTFIELDS => json_encode([
                'domain' => $domain,
                'auth_code' => $eppCode,
                'period' => $years,
            ])
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);
        
        if ($httpCode === 200 && isset($data['status']) && $data['status'] === 'success') {
            return [
                'success' => true,
                'domain' => $domain,
                'transaction_id' => $data['transaction_id'] ?? '',
                'status' => $data['transfer_status'] ?? 'pending',
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Domain transfer failed',
        ];
    }

    public function getDomainInfo($domain)
    {
        // Get domain information including nameservers and expiry
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->baseUrl . '/domains/' . $domain,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->apiKey,
            ]
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);
        
        if ($httpCode === 200 && isset($data['domain'])) {
            return [
                'success' => true,
                'domain' => $data['domain'],
                'status' => $data['status'],
                'expiration_date' => $data['expiration_date'],
                'nameservers' => $data['nameservers'] ?? [],
            ];
        }

        return [
            'success' => false,
            'error' => 'Failed to retrieve domain information',
        ];
    }
}
