<?php

namespace App\Modules\PaymentGateways\Flutterwave;

class FlutterwaveGateway
{
    protected $config;
    protected $secretKey;
    protected $publicKey;

    public function __construct($config = [])
    {
        $this->config = $config;
        $this->secretKey = $config['secret_key'] ?? env('FLUTTERWAVE_SECRET');
        $this->publicKey = $config['public_key'] ?? env('FLUTTERWAVE_PUBLIC');
    }

    public function createPayment($invoice)
    {
        // Real Flutterwave API Standard Payment
        // Documentation: https://developer.flutterwave.com/docs/collecting-payments/standard
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://api.flutterwave.com/v3/payments',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->secretKey,
                'Content-Type: application/json'
            ],
            CURLOPT_POSTFIELDS => json_encode([
                'tx_ref' => 'INV-' . $invoice->id . '-' . time(),
                'amount' => $invoice->total,
                'currency' => strtoupper($invoice->currency ?? 'NGN'),
                'redirect_url' => url('/payment/flutterwave/callback'),
                'customer' => [
                    'email' => $invoice->user->email,
                    'name' => $invoice->user->name,
                ],
                'customizations' => [
                    'title' => 'Invoice Payment',
                    'description' => 'Payment for Invoice #' . $invoice->id,
                    'logo' => asset('logo.png'),
                ],
                'meta' => [
                    'invoice_id' => $invoice->id,
                    'user_id' => $invoice->user_id,
                ]
            ])
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);

        if ($httpCode === 200 && $data['status'] === 'success') {
            return [
                'success' => true,
                'payment_link' => $data['data']['link'],
                'transaction_id' => $data['data']['tx_ref'],
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Payment creation failed',
        ];
    }

    public function verifyPayment($transactionId)
    {
        // Real Flutterwave API Transaction Verification
        // Documentation: https://developer.flutterwave.com/docs/collecting-payments/transactions
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://api.flutterwave.com/v3/transactions/' . $transactionId . '/verify',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->secretKey,
            ]
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);

        if ($httpCode === 200 && $data['status'] === 'success') {
            $transactionData = $data['data'];
            return [
                'success' => $transactionData['status'] === 'successful',
                'status' => $transactionData['status'],
                'amount' => $transactionData['amount'],
                'currency' => $transactionData['currency'],
                'tx_ref' => $transactionData['tx_ref'],
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Verification failed',
        ];
    }

    public function refundPayment($transactionId, $amount = null)
    {
        // Real Flutterwave API Refund
        // Documentation: https://developer.flutterwave.com/docs/refunds
        
        $curl = curl_init();
        
        $postData = ['id' => $transactionId];
        if ($amount) {
            $postData['amount'] = $amount;
        }
        
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://api.flutterwave.com/v3/transactions/' . $transactionId . '/refund',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->secretKey,
                'Content-Type: application/json'
            ],
            CURLOPT_POSTFIELDS => json_encode($postData)
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);

        if ($httpCode === 200 && $data['status'] === 'success') {
            return [
                'success' => true,
                'refund_id' => $data['data']['id'],
                'amount' => $data['data']['amount'],
                'status' => $data['data']['status'],
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Refund failed',
        ];
    }

    public function handleWebhook($payload, $signature)
    {
        // Real Flutterwave webhook signature verification
        // Documentation: https://developer.flutterwave.com/docs/integration-guides/webhooks/
        
        $webhookSecret = $this->config['webhook_secret'] ?? env('FLUTTERWAVE_WEBHOOK_SECRET');
        
        $expectedSignature = hash_hmac('sha256', $payload, $webhookSecret);
        
        if ($signature === $expectedSignature) {
            $event = json_decode($payload, true);
            
            return [
                'success' => true,
                'event' => $event['event'],
                'data' => $event['data'],
            ];
        }

        return [
            'success' => false,
            'error' => 'Invalid signature',
        ];
    }
}
