<?php

namespace App\Modules\PaymentGateways\Razorpay;

class RazorpayGateway
{
    protected $config;
    protected $keyId;
    protected $keySecret;

    public function __construct($config = [])
    {
        $this->config = $config;
        $this->keyId = $config['key_id'] ?? env('RAZORPAY_KEY_ID');
        $this->keySecret = $config['key_secret'] ?? env('RAZORPAY_KEY_SECRET');
    }

    public function createPayment($invoice)
    {
        // Real Razorpay API Order creation
        // Documentation: https://razorpay.com/docs/api/orders/
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://api.razorpay.com/v1/orders',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_USERPWD => $this->keyId . ':' . $this->keySecret,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json'
            ],
            CURLOPT_POSTFIELDS => json_encode([
                'amount' => round($invoice->total * 100), // Razorpay uses paise
                'currency' => strtoupper($invoice->currency ?? 'INR'),
                'receipt' => 'INV-' . $invoice->id,
                'notes' => [
                    'invoice_id' => $invoice->id,
                    'user_id' => $invoice->user_id,
                ]
            ])
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);

        if ($httpCode === 200 && isset($data['id'])) {
            return [
                'success' => true,
                'order_id' => $data['id'],
                'amount' => $data['amount'],
                'currency' => $data['currency'],
                'transaction_id' => $data['id'],
            ];
        }

        return [
            'success' => false,
            'error' => $data['error']['description'] ?? 'Order creation failed',
        ];
    }

    public function verifyPayment($paymentId, $orderId, $signature)
    {
        // Real Razorpay payment signature verification
        // Documentation: https://razorpay.com/docs/payments/server-integration/php/payment-gateway/build-integration/
        
        $expectedSignature = hash_hmac('sha256', $orderId . '|' . $paymentId, $this->keySecret);
        
        if ($signature === $expectedSignature) {
            // Fetch payment details
            $curl = curl_init();
            
            curl_setopt_array($curl, [
                CURLOPT_URL => 'https://api.razorpay.com/v1/payments/' . $paymentId,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_USERPWD => $this->keyId . ':' . $this->keySecret,
            ]);

            $response = curl_exec($curl);
            $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            curl_close($curl);

            $data = json_decode($response, true);

            if ($httpCode === 200) {
                return [
                    'success' => $data['status'] === 'captured',
                    'status' => $data['status'],
                    'amount' => $data['amount'] / 100,
                    'currency' => $data['currency'],
                ];
            }
        }

        return [
            'success' => false,
            'error' => 'Invalid signature or payment not found',
        ];
    }

    public function refundPayment($paymentId, $amount = null)
    {
        // Real Razorpay API Refund
        // Documentation: https://razorpay.com/docs/api/refunds/
        
        $curl = curl_init();
        
        $postData = [];
        if ($amount) {
            $postData['amount'] = round($amount * 100);
        }
        
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://api.razorpay.com/v1/payments/' . $paymentId . '/refund',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_USERPWD => $this->keyId . ':' . $this->keySecret,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json'
            ],
            CURLOPT_POSTFIELDS => json_encode($postData)
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);

        if ($httpCode === 200 && isset($data['id'])) {
            return [
                'success' => true,
                'refund_id' => $data['id'],
                'amount' => $data['amount'] / 100,
                'status' => $data['status'],
            ];
        }

        return [
            'success' => false,
            'error' => $data['error']['description'] ?? 'Refund failed',
        ];
    }

    public function handleWebhook($payload, $signature)
    {
        // Real Razorpay webhook signature verification
        // Documentation: https://razorpay.com/docs/webhooks/
        
        $webhookSecret = $this->config['webhook_secret'] ?? env('RAZORPAY_WEBHOOK_SECRET');
        
        $expectedSignature = hash_hmac('sha256', $payload, $webhookSecret);
        
        if ($signature === $expectedSignature) {
            $event = json_decode($payload, true);
            
            return [
                'success' => true,
                'event' => $event['event'],
                'entity' => $event['payload']['payment']['entity'],
            ];
        }

        return [
            'success' => false,
            'error' => 'Invalid signature',
        ];
    }
}
