# 🚀 BattleMania Deployment Guide

This comprehensive guide covers deploying BattleMania to any hosting platform including shared hosting, VPS, and cloud providers.

## 📋 Pre-deployment Checklist

### Environment Configuration
- [ ] Set `APP_ENV=production`
- [ ] Set `APP_DEBUG=false`
- [ ] Generate secure `APP_KEY`
- [ ] Configure production database
- [ ] Set up mail server
- [ ] Configure file storage (local or cloud)
- [ ] Set secure admin password
- [ ] Configure payment gateways through admin panel

### Security Requirements
- [ ] HTTPS enabled with valid SSL certificate
- [ ] Database password complexity
- [ ] Session encryption enabled
- [ ] Rate limiting configured
- [ ] Server firewall configured

## 🌐 cPanel Shared Hosting Setup

### 1. File Upload
```bash
# Download and extract project files
# Upload all files to public_html or subdomain folder
```

### 2. Database Setup
1. Create MySQL database through cPanel
2. Create database user with full permissions
3. Note down database credentials

### 3. Environment Configuration
```env
# Create .env file in root directory
APP_NAME="BattleMania"
APP_ENV=production
APP_DEBUG=false
APP_URL=https://yourdomain.com

DB_CONNECTION=mysql
DB_HOST=localhost
DB_PORT=3306
DB_DATABASE=your_database_name
DB_USERNAME=your_database_user
DB_PASSWORD=your_database_password

MAIL_MAILER=smtp
MAIL_HOST=your_mail_host
MAIL_PORT=587
MAIL_USERNAME=your_email
MAIL_PASSWORD=your_email_password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=noreply@yourdomain.com
MAIL_FROM_NAME="BattleMania"

SESSION_DRIVER=file
CACHE_DRIVER=file
QUEUE_CONNECTION=database

# Generate this using: php artisan key:generate
APP_KEY=base64:your_generated_key_here
```

### 4. File Permissions
```bash
# Set correct permissions
chmod 755 -R public_html/
chmod 775 -R storage/
chmod 775 -R bootstrap/cache/
```

### 5. Database Migration (via File Manager terminal or SSH)
```bash
# Navigate to project directory
cd public_html

# Install dependencies (if composer is available)
composer install --no-dev --optimize-autoloader

# Run migrations
php artisan migrate --force

# Seed database
php artisan db:seed --force

# Optimize for production
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

### 6. Web Directory Setup
If your hosting requires the Laravel public folder as document root:
1. Move contents of `public/` to your domain's document root
2. Update `index.php` to point to correct paths:
```php
require __DIR__.'/../vendor/autoload.php';
$app = require_once __DIR__.'/../bootstrap/app.php';
```

## 🐧 Ubuntu/Debian VPS Setup

### 1. Server Requirements
```bash
# Update system
sudo apt update && sudo apt upgrade -y

# Install required packages
sudo apt install -y nginx mysql-server php8.3-fpm php8.3-mysql php8.3-xml php8.3-curl php8.3-zip php8.3-mbstring php8.3-bcmath php8.3-gd php8.3-redis unzip git composer
```

### 2. MySQL Setup
```bash
# Secure MySQL installation
sudo mysql_secure_installation

# Create database and user
sudo mysql -u root -p
```

```sql
CREATE DATABASE battlemania_production;
CREATE USER 'battlemania_user'@'localhost' IDENTIFIED BY 'secure_random_password';
GRANT ALL PRIVILEGES ON battlemania_production.* TO 'battlemania_user'@'localhost';
FLUSH PRIVILEGES;
EXIT;
```

### 3. Nginx Configuration
```bash
# Create site configuration
sudo nano /etc/nginx/sites-available/battlemania
```

```nginx
server {
    listen 80;
    server_name yourdomain.com www.yourdomain.com;
    root /var/www/battlemania/public;
    index index.php index.html;

    # Security headers
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;

    # Gzip compression
    gzip on;
    gzip_vary on;
    gzip_types text/plain text/css application/json application/javascript text/xml application/xml text/javascript;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_split_path_info ^(.+\.php)(/.+)$;
        fastcgi_pass unix:/var/run/php/php8.3-fpm.sock;
        fastcgi_index index.php;
        include fastcgi_params;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
    }

    # Block access to sensitive files
    location ~ /\. {
        deny all;
    }

    location ~ /storage {
        deny all;
    }

    location ~ /bootstrap/cache {
        deny all;
    }
}
```

```bash
# Enable site and restart services
sudo ln -s /etc/nginx/sites-available/battlemania /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl restart nginx
sudo systemctl restart php8.3-fpm
```

### 4. Application Deployment
```bash
# Create directory
sudo mkdir -p /var/www/battlemania
cd /var/www/battlemania

# Clone repository
sudo git clone https://github.com/your-username/battlemania.git .

# Set permissions
sudo chown -R www-data:www-data /var/www/battlemania
sudo chmod -R 755 /var/www/battlemania
sudo chmod -R 775 /var/www/battlemania/storage
sudo chmod -R 775 /var/www/battlemania/bootstrap/cache

# Install dependencies
sudo -u www-data composer install --no-dev --optimize-autoloader

# Configure environment
sudo -u www-data cp .env.example .env
sudo -u www-data php artisan key:generate

# Edit environment file
sudo nano .env
```

Update `.env` with production values, then:

```bash
# Run migrations and seeders
sudo -u www-data php artisan migrate:fresh --seed --force

# Cache configuration for performance
sudo -u www-data php artisan config:cache
sudo -u www-data php artisan route:cache
sudo -u www-data php artisan view:cache
```

## ☁️ Cloud Hosting (AWS, DigitalOcean, etc.)

### 1. Instance Setup
- Launch Ubuntu 20.04+ instance
- Configure security groups (ports 22, 80, 443)
- Set up SSH key authentication
- Follow VPS setup instructions above

### 2. Load Balancer Configuration (if using multiple servers)
```nginx
upstream battlemania_app {
    server 10.0.1.10:80;
    server 10.0.1.11:80;
}

server {
    listen 80;
    server_name yourdomain.com;
    
    location / {
        proxy_pass http://battlemania_app;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
}
```

### 3. Database (RDS/Managed Database)
```env
# Use managed database service
DB_HOST=your-rds-endpoint.amazonaws.com
DB_PORT=3306
DB_DATABASE=battlemania
DB_USERNAME=admin
DB_PASSWORD=secure_password
```

## 🔒 SSL Certificate Setup

### Using Let's Encrypt (Free)
```bash
# Install Certbot
sudo apt install certbot python3-certbot-nginx

# Generate certificate
sudo certbot --nginx -d yourdomain.com -d www.yourdomain.com

# Test auto-renewal
sudo certbot renew --dry-run
```

### Using Cloudflare (Recommended)
1. Add your domain to Cloudflare
2. Update nameservers
3. Enable "Full (Strict)" SSL mode
4. Enable security features (rate limiting, bot protection)

## ⚙️ Platform Configuration

### 1. Admin Panel Setup
1. Visit `/admin` after deployment
2. Login with admin credentials (admin@battlemania.com / password)
3. Navigate to Settings to configure:

#### General Settings
- Platform name and description
- Contact information
- Timezone and currency

#### Payment Gateways
- Add Paystack credentials
- Configure Flutterwave
- Set up PayPal/Stripe if needed

#### Security Settings
- Set rate limits
- Configure session timeouts
- Enable 2FA for admins

#### Platform Configuration
- Set commission rates
- Configure deposit/withdrawal limits
- Enable KYC requirements

#### Feature Settings
- Enable/disable platform features
- Configure referral system
- Manage tournament settings

### 2. Email/SMS Configuration
Navigate to Admin → APIs → Notifications:
- Configure email providers (Resend, Mailgun, SendGrid)
- Set up SMS providers (Termii, Twilio, Nexmo)
- Test configurations

### 3. Content Management
- Create blog posts and announcements
- Set up popup notifications for users
- Manage tournament highlights

## 📊 Monitoring & Maintenance

### 1. Log Management
```bash
# Set up log rotation
sudo nano /etc/logrotate.d/battlemania
```

```
/var/www/battlemania/storage/logs/*.log {
    daily
    missingok
    rotate 30
    compress
    notifempty
    create 0644 www-data www-data
}
```

### 2. Backup Script
```bash
# Create backup script
sudo nano /usr/local/bin/battlemania-backup.sh
```

```bash
#!/bin/bash
BACKUP_DIR="/var/backups/battlemania"
DATE=$(date +%Y%m%d_%H%M%S)

# Create backup directory
mkdir -p $BACKUP_DIR

# Database backup
mysqldump -u battlemania_user -p'secure_password' battlemania_production > $BACKUP_DIR/db_backup_$DATE.sql

# Application backup
tar -czf $BACKUP_DIR/app_backup_$DATE.tar.gz /var/www/battlemania --exclude=/var/www/battlemania/storage/logs

# Remove backups older than 7 days
find $BACKUP_DIR -name "*.sql" -mtime +7 -delete
find $BACKUP_DIR -name "*.tar.gz" -mtime +7 -delete
```

```bash
# Make executable and schedule
sudo chmod +x /usr/local/bin/battlemania-backup.sh
sudo crontab -e

# Add to crontab (daily backup at 2 AM)
0 2 * * * /usr/local/bin/battlemania-backup.sh
```

### 3. Performance Monitoring
```bash
# Install monitoring tools
sudo apt install htop iotop nethogs

# Monitor Laravel queues (if using)
sudo -u www-data php artisan queue:work --daemon
```

## 🚀 Deployment Automation

### Using GitHub Actions
Create `.github/workflows/deploy.yml`:

```yaml
name: Deploy to Production

on:
  push:
    branches: [main]

jobs:
  deploy:
    runs-on: ubuntu-latest
    
    steps:
    - uses: actions/checkout@v2
    
    - name: Deploy to server
      uses: appleboy/ssh-action@v0.1.5
      with:
        host: ${{ secrets.HOST }}
        username: ${{ secrets.USERNAME }}
        key: ${{ secrets.SSH_KEY }}
        script: |
          cd /var/www/battlemania
          git pull origin main
          sudo -u www-data composer install --no-dev --optimize-autoloader
          sudo -u www-data php artisan migrate --force
          sudo -u www-data php artisan config:cache
          sudo -u www-data php artisan route:cache
          sudo -u www-data php artisan view:cache
          sudo systemctl restart php8.3-fpm
```

## 🛡️ Security Hardening

### 1. Server Security
```bash
# Disable root login
sudo nano /etc/ssh/sshd_config
# Set: PermitRootLogin no

# Configure firewall
sudo ufw allow OpenSSH
sudo ufw allow 'Nginx Full'
sudo ufw enable

# Install fail2ban
sudo apt install fail2ban
sudo systemctl enable fail2ban
```

### 2. Application Security
```bash
# Set strict file permissions
sudo find /var/www/battlemania -type f -exec chmod 644 {} \;
sudo find /var/www/battlemania -type d -exec chmod 755 {} \;
sudo chmod -R 775 /var/www/battlemania/storage
sudo chmod -R 775 /var/www/battlemania/bootstrap/cache
sudo chmod 644 /var/www/battlemania/.env
```

## 📈 Performance Optimization

### 1. PHP OPcache
```bash
# Edit PHP configuration
sudo nano /etc/php/8.3/fpm/php.ini

# Enable OPcache
opcache.enable=1
opcache.memory_consumption=256
opcache.max_accelerated_files=7963
opcache.validate_timestamps=0
```

### 2. Redis Cache (Optional)
```bash
# Install Redis
sudo apt install redis-server

# Update .env
CACHE_DRIVER=redis
SESSION_DRIVER=redis
QUEUE_CONNECTION=redis
```

### 3. Database Optimization
```bash
# MySQL configuration
sudo nano /etc/mysql/mysql.conf.d/mysqld.cnf

# Add optimizations
innodb_buffer_pool_size = 256M
query_cache_type = 1
query_cache_size = 64M
```

## 🔧 Troubleshooting

### Common Issues

1. **Permission Errors**
   ```bash
   sudo chown -R www-data:www-data /var/www/battlemania
   sudo chmod -R 775 storage bootstrap/cache
   ```

2. **Database Connection Issues**
   ```bash
   # Check MySQL status
   sudo systemctl status mysql
   
   # Test connection
   mysql -u battlemania_user -p battlemania_production
   ```

3. **Nginx 502 Errors**
   ```bash
   # Check PHP-FPM status
   sudo systemctl status php8.3-fpm
   
   # Check error logs
   sudo tail -f /var/log/nginx/error.log
   ```

4. **Storage Issues**
   ```bash
   # Clear caches
   sudo -u www-data php artisan cache:clear
   sudo -u www-data php artisan config:clear
   sudo -u www-data php artisan view:clear
   ```

## 🎯 Post-Deployment Checklist

- [ ] Admin panel accessible and configured
- [ ] Payment gateways tested and working
- [ ] Email/SMS notifications configured
- [ ] SSL certificate installed and working
- [ ] Database backups scheduled
- [ ] Monitoring systems in place
- [ ] Security measures implemented
- [ ] Performance optimizations applied
- [ ] Error logging configured
- [ ] Domain DNS properly configured

## 📞 Support

For deployment support:
- Check the logs: `/var/www/battlemania/storage/logs/laravel.log`
- Nginx logs: `/var/log/nginx/error.log`
- PHP-FPM logs: `/var/log/php8.3-fpm.log`
- Admin panel settings for configuration issues

## 🎮 Default Access

After deployment:
- **Admin Panel**: `https://yourdomain.com/admin`
- **Admin Account**: admin@battlemania.com / password
- **Demo Users**: gamer1@example.com / password, player2@example.com / password

**Important**: Change default passwords immediately after deployment!

---

**Ready for gaming! 🎮 Your BattleMania platform is now live and fully functional.**