<?php

namespace App\Enums;

enum UserRole: string
{
    case BASIC_USER = 'basic_user';
    case PREMIUM_USER = 'premium_user';
    case CREATOR = 'creator';
    case MODERATOR = 'moderator';
    case ADMIN = 'admin';

    /**
     * Get display name for the role
     */
    public function getDisplayName(): string
    {
        return match($this) {
            self::BASIC_USER => 'Basic User',
            self::PREMIUM_USER => 'Premium User',
            self::CREATOR => 'Creator',
            self::MODERATOR => 'Moderator',
            self::ADMIN => 'Admin',
        };
    }

    /**
     * Get description for the role
     */
    public function getDescription(): string
    {
        return match($this) {
            self::BASIC_USER => 'Standard user account with basic features (Level 1)',
            self::PREMIUM_USER => 'Premium user with advanced features (Level 29)',
            self::CREATOR => 'Content creator with blog posting abilities (Creator Level)',
            self::MODERATOR => 'Moderator with dispute management capabilities',
            self::ADMIN => 'Full administrator with all system privileges',
        };
    }

    /**
     * Get the level that should be assigned when this role is set
     */
    public function getAutoAssignedLevel(): ?int
    {
        return match($this) {
            self::BASIC_USER => 1,
            self::PREMIUM_USER => 29,
            self::CREATOR => null, // Creator level ID will be determined dynamically
            self::MODERATOR => null, // Level doesn't change for moderators
            self::ADMIN => null, // Admin level doesn't change
        };
    }

    /**
     * Check if this role can manage disputes
     */
    public function canManageDisputes(): bool
    {
        return in_array($this, [self::MODERATOR, self::ADMIN]);
    }

    /**
     * Check if this role has administrative privileges
     */
    public function isAdmin(): bool
    {
        return $this === self::ADMIN;
    }

    /**
     * Check if this role can create content/blogs
     */
    public function canCreateContent(): bool
    {
        return in_array($this, [self::CREATOR, self::ADMIN]);
    }

    /**
     * Get all role values as array
     */
    public static function values(): array
    {
        return array_column(self::cases(), 'value');
    }

    /**
     * Get roles for dropdown selection
     */
    public static function forDropdown(): array
    {
        $options = [];
        foreach (self::cases() as $role) {
            $options[$role->value] = $role->getDisplayName();
        }
        return $options;
    }
}