<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Challenge;
use App\Models\Game;
use App\Models\User;
use App\Services\WalletService;
use Illuminate\Http\Request;

class ChallengeController extends Controller
{
    protected $walletService;

    public function __construct(WalletService $walletService)
    {
        $this->walletService = $walletService;
    }

    public function index(Request $request)
    {
        $query = Challenge::with(['creator', 'accepter', 'game']);

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->game_id) {
            $query->where('game_id', $request->game_id);
        }

        $challenges = $query->orderBy('created_at', 'desc')->paginate(15);
        $games = Game::orderBy('name')->get();

        return view('admin.challenges.index', compact('challenges', 'games'));
    }

    public function create()
    {
        $games = Game::orderBy('name')->get();
        $users = User::where('is_banned', false)->orderBy('username')->get();

        return view('admin.challenges.create', compact('games', 'users'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'creator_id' => 'required|exists:users,id',
            'accepter_id' => 'nullable|exists:users,id|different:creator_id',
            'game_id' => 'required|exists:games,id',
            'wager_amount' => 'required|numeric|min:0.01',
            'best_of' => 'required|integer|min:1|max:7',
            'rules_text' => 'required|string|max:1000',
        ]);

        $creator = User::findOrFail($request->creator_id);

        // Check if creator has enough balance
        if ($creator->wallet_balance < $request->wager_amount) {
            return redirect()->back()->with('error', 'Creator does not have sufficient balance.');
        }

        // Deduct wager from creator
        $this->walletService->deductBalance($creator, $request->wager_amount, 'challenge_wager',
                                           'Challenge wager deduction');

        $challenge = Challenge::create([
            'creator_id' => $request->creator_id,
            'game_id' => $request->game_id,
            'wager_amount' => $request->wager_amount,
            'best_of' => $request->best_of,
            'rules_text' => $request->rules_text,
            'status' => 'open',
        ]);

        // If accepter is specified, auto-accept the challenge
        if ($request->accepter_id) {
            $accepter = User::findOrFail($request->accepter_id);

            if ($accepter->wallet_balance < $request->wager_amount) {
                return redirect()->back()->with('error', 'Accepter does not have sufficient balance.');
            }

            // Deduct wager from accepter
            $this->walletService->deductBalance($accepter, $request->wager_amount, 'challenge_wager',
                                               'Challenge wager deduction');

            $challenge->update([
                'accepter_id' => $request->accepter_id,
                'status' => 'accepted',
            ]);
        }

        return redirect()->route('admin.challenges.index')
                        ->with('success', 'Challenge created successfully!');
    }

    public function show(Challenge $challenge)
    {
        $challenge->load(['creator', 'accepter', 'game', 'matchProofs', 'dispute']);
        
        return view('admin.challenges.show', compact('challenge'));
    }

    public function hookUsers(Request $request)
    {
        $request->validate([
            'player1_id' => 'required|exists:users,id',
            'player2_id' => 'required|exists:users,id|different:player1_id',
            'game_id' => 'required|exists:games,id',
            'wager_amount' => 'required|numeric|min:0.01',
            'best_of' => 'required|integer|min:1|max:7',
            'rules_text' => 'required|string|max:1000',
        ]);

        $player1 = User::findOrFail($request->player1_id);
        $player2 = User::findOrFail($request->player2_id);

        // Check if both players have enough balance
        if ($player1->wallet_balance < $request->wager_amount) {
            return redirect()->back()->with('error', $player1->username . ' does not have sufficient balance.');
        }

        if ($player2->wallet_balance < $request->wager_amount) {
            return redirect()->back()->with('error', $player2->username . ' does not have sufficient balance.');
        }

        // Deduct wagers from both players
        $this->walletService->deductBalance($player1, $request->wager_amount, 'challenge_wager',
                                           'Admin created challenge wager');
        $this->walletService->deductBalance($player2, $request->wager_amount, 'challenge_wager',
                                           'Admin created challenge wager');

        Challenge::create([
            'creator_id' => $request->player1_id,
            'accepter_id' => $request->player2_id,
            'game_id' => $request->game_id,
            'wager_amount' => $request->wager_amount,
            'best_of' => $request->best_of,
            'rules_text' => $request->rules_text,
            'status' => 'accepted',
        ]);

        return redirect()->route('admin.challenges.index')
                        ->with('success', 'Challenge created and both players matched successfully!');
    }

    public function resolveChallenge(Request $request, Challenge $challenge)
    {
        $request->validate([
            'winner' => 'required|in:creator,accepter,draw',
        ]);

        if ($challenge->status !== 'accepted' && $challenge->status !== 'disputed') {
            return redirect()->back()->with('error', 'Challenge cannot be resolved in its current state.');
        }

        $totalPrize = $challenge->wager_amount * 2;
        $platformFee = $totalPrize * 0.05; // 5% platform fee
        $winnerPrize = $totalPrize - $platformFee;

        if ($request->winner === 'creator') {
            $this->walletService->addBalance($challenge->creator, $winnerPrize, 'challenge_win',
                                            'Challenge victory prize');
            $challenge->update(['result' => 'creator_win', 'status' => 'completed']);
        } elseif ($request->winner === 'accepter') {
            $this->walletService->addBalance($challenge->accepter, $winnerPrize, 'challenge_win',
                                            'Challenge victory prize');
            $challenge->update(['result' => 'accepter_win', 'status' => 'completed']);
        } else {
            // Draw - refund both players
            $this->walletService->addBalance($challenge->creator, $challenge->wager_amount, 'challenge_refund',
                                            'Challenge draw refund');
            $this->walletService->addBalance($challenge->accepter, $challenge->wager_amount, 'challenge_refund',
                                            'Challenge draw refund');
            $challenge->update(['result' => 'draw', 'status' => 'completed']);
        }

        return redirect()->back()->with('success', 'Challenge resolved successfully!');
    }

    public function cancelChallenge(Challenge $challenge)
    {
        if ($challenge->status === 'completed' || $challenge->status === 'cancelled') {
            return redirect()->back()->with('error', 'Challenge is already resolved.');
        }

        // Refund wagers
        $this->walletService->addBalance($challenge->creator, $challenge->wager_amount, 'challenge_refund',
                                        'Challenge cancellation refund');

        if ($challenge->accepter) {
            $this->walletService->addBalance($challenge->accepter, $challenge->wager_amount, 'challenge_refund',
                                            'Challenge cancellation refund');
        }

        $challenge->update(['status' => 'cancelled']);

        return redirect()->back()->with('success', 'Challenge cancelled and refunded successfully!');
    }
}