<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CreatorRequestController extends Controller
{
    /**
     * Display a listing of Creator requests.
     */
    public function index()
    {
        $this->authorize('admin-actions'); // Ensure only admins can access
        
        $pendingRequests = User::where('creator_request_status', 'pending')
            ->orderBy('creator_request_date', 'desc')
            ->get();
            
        $recentRequests = User::whereIn('creator_request_status', ['approved', 'rejected'])
            ->orderBy('updated_at', 'desc')
            ->limit(20)
            ->get();

        return view('admin.creator-requests.index', compact('pendingRequests', 'recentRequests'));
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        $this->authorize('admin-actions');
        
        if (!$user->hasPendingCreatorRequest()) {
            return redirect()->back()->with('error', 'User does not have a pending Creator request.');
        }

        return view('admin.creator-requests.show', compact('user'));
    }

    /**
     * Approve a Creator request.
     */
    public function approve(User $user)
    {
        $this->authorize('admin-actions');

        if (!$user->hasPendingCreatorRequest()) {
            return redirect()->back()->with('error', 'User does not have a pending Creator request.');
        }

        $user->approveCreatorRequest();

        return redirect()->route('admin.creator-requests.index')
            ->with('success', "Creator request for {$user->username} has been approved.");
    }

    /**
     * Reject a Creator request.
     */
    public function reject(Request $request, User $user)
    {
        $this->authorize('admin-actions');

        if (!$user->hasPendingCreatorRequest()) {
            return redirect()->back()->with('error', 'User does not have a pending Creator request.');
        }

        $user->rejectCreatorRequest();

        return redirect()->route('admin.creator-requests.index')
            ->with('success', "Creator request for {$user->username} has been rejected.");
    }

    /**
     * Manually assign Creator status to a user (admin feature)
     */
    public function manualAssign(Request $request)
    {
        $this->authorize('admin-actions');

        $request->validate([
            'user_id' => 'required|exists:users,id',
        ]);

        $user = User::findOrFail($request->user_id);

        if ($user->isCreator() || $user->isAdmin()) {
            return redirect()->back()->with('error', 'User is already Creator or Admin.');
        }

        // Check if user meets minimum requirements (e.g., games played)
        if (method_exists($user, 'meetsCreatorRequirements')) {
            if (!$user->meetsCreatorRequirements()) {
                return redirect()->back()->with('error', 'User does not meet the minimum requirements for Creator status.');
            }
        } else {
            // Fallback: check for a games_played attribute (replace 10 with actual requirement)
            if (property_exists($user, 'games_played') && $user->games_played < 10) {
                return redirect()->back()->with('error', 'User does not meet the minimum requirements for Creator status.');
            }
        }
        // Directly assign Creator role without request
        $user->update([
            'role' => 'creator',
            'creator_request_status' => 'approved',
            'creator_request_date' => now(),
            'creator_request_reason' => 'Manually assigned by admin: ' . Auth::user()->username,
        ]);

        return redirect()->back()->with('success', "Creator status manually assigned to {$user->username}.");
    }
}
