<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Dispute;
use App\Services\WalletService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class DisputeController extends Controller
{
    public function __construct(
        private WalletService $walletService
    ) {}

    /**
     * Display a listing of disputes.
     */
    public function index()
    {
        $disputes = Dispute::with(['challenge.creator', 'challenge.accepter', 'challenge.game', 'openedBy', 'moderator'])
            ->latest()
            ->paginate(20);

        return view('admin.disputes.index', compact('disputes'));
    }

    /**
     * Display the specified dispute.
     */
    public function show(Dispute $dispute)
    {
        $dispute->load([
            'challenge.creator',
            'challenge.accepter',
            'challenge.game',
            'challenge.matchProofs.user',
            'openedBy',
            'moderator'
        ]);

        return view('admin.disputes.show', compact('dispute'));
    }

    /**
     * Resolve a dispute.
     */
    public function decide(Request $request, Dispute $dispute)
    {
        $request->validate([
            'decision' => 'required|in:creator_win,accepter_win,refund',
            'resolution' => 'required|string|max:1000',
        ]);

        $challenge = $dispute->challenge;

        // Update dispute
        $dispute->update([
            'status' => 'resolved',
            'resolution' => $request->resolution,
            'admin_id' => Auth::id(),
        ]);

        // Update challenge based on decision
        if ($request->decision === 'refund') {
            $challenge->update([
                'status' => 'completed',
                'result' => 'none',
            ]);
            
            // Refund both players
            $this->walletService->refundChallenge($challenge);
        } else {
            $challenge->update([
                'status' => 'completed',
                'result' => $request->decision,
            ]);
            
            // Process payout to winner
            $this->walletService->processChallengeWin($challenge);
        }

        return redirect()->route('admin.disputes.index')
            ->with('success', 'Dispute resolved successfully!');
    }
}