<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Game;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class GameController extends Controller
{
    public function index()
    {
        $games = Game::withCount(['challenges', 'tournaments'])
                    ->orderBy('name')
                    ->paginate(15);

        return view('admin.games.index', compact('games'));
    }

    public function create()
    {
        return view('admin.games.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:games',
            'platform' => 'required|string|max:100',
        ]);

        Game::create([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'platform' => $request->platform,
        ]);

        return redirect()->route('admin.games.index')
                        ->with('success', 'Game created successfully!');
    }

    public function show(Game $game)
    {
        $game->load(['challenges.creator', 'challenges.accepter', 'tournaments.host']);
        
        return view('admin.games.show', compact('game'));
    }

    public function edit(Game $game)
    {
        return view('admin.games.edit', compact('game'));
    }

    public function update(Request $request, Game $game)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:games,name,' . $game->id,
            'platform' => 'required|string|max:100',
        ]);

        $game->update([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'platform' => $request->platform,
        ]);

        return redirect()->route('admin.games.index')
                        ->with('success', 'Game updated successfully!');
    }

    public function destroy(Game $game)
    {
        if ($game->challenges()->count() > 0 || $game->tournaments()->count() > 0) {
            return redirect()->route('admin.games.index')
                            ->with('error', 'Cannot delete game with existing challenges or tournaments.');
        }

        $game->delete();

        return redirect()->route('admin.games.index')
                        ->with('success', 'Game deleted successfully!');
    }
}