<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\NotificationProvider;
use App\Models\NotificationSetting;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class NotificationController extends Controller
{
    protected $notificationService;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * Display notification settings dashboard.
     */
    public function index()
    {
        $emailProviders = NotificationProvider::byType('email')->get();
        $smsProviders = NotificationProvider::byType('sms')->get();
        
        $settings = [
            'email_enabled' => NotificationSetting::get('email_enabled', true),
            'sms_enabled' => NotificationSetting::get('sms_enabled', true),
            'default_email_from' => NotificationSetting::get('default_email_from', 'noreply@battlemania.com'),
            'test_email' => NotificationSetting::get('test_email', 'admin@battlemania.com'),
            'test_phone' => NotificationSetting::get('test_phone', '+1234567890'),
        ];

        return view('admin.notifications.index', compact('emailProviders', 'smsProviders', 'settings'));
    }

    /**
     * Show provider configuration form.
     */
    public function createProvider(Request $request)
    {
        $type = $request->get('type', 'email');
        $name = $request->get('name');

        $availableProviders = $this->getAvailableProviders($type);

        if ($name && !isset($availableProviders[$name])) {
            abort(404, 'Provider not found');
        }

        $provider = $name ? new NotificationProvider([
            'name' => $name,
            'type' => $type,
            'display_name' => $availableProviders[$name]['display_name'],
        ]) : null;

        return view('admin.notifications.create-provider', compact('provider', 'type', 'availableProviders'));
    }

    /**
     * Store a new provider configuration.
     */
    public function storeProvider(Request $request)
    {
        $availableProviders = $this->getAvailableProviders($request->type);
        
        if (!isset($availableProviders[$request->name])) {
            return back()->withErrors(['name' => 'Invalid provider selected']);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string',
            'type' => 'required|in:email,sms',
            'display_name' => 'required|string|max:255',
            'priority' => 'required|integer|min:1|max:10',
            'configuration' => 'required|array',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Check if provider already exists
        $existing = NotificationProvider::where('name', $request->name)
            ->where('type', $request->type)
            ->first();

        if ($existing) {
            return back()->withErrors(['name' => 'This provider is already configured']);
        }

        // Validate configuration fields
        $provider = new NotificationProvider([
            'name' => $request->name,
            'type' => $request->type,
            'configuration' => $request->configuration,
        ]);

        $requiredFields = $provider->getRequiredConfigurationFields();
        foreach ($requiredFields as $field) {
            if (!isset($request->configuration[$field]) || empty($request->configuration[$field])) {
                return back()->withErrors(["configuration.{$field}" => "The {$field} field is required"]);
            }
        }

        $provider = NotificationProvider::create([
            'name' => $request->name,
            'type' => $request->type,
            'display_name' => $request->display_name,
            'configuration' => $request->configuration,
            'priority' => $request->priority,
            'is_active' => false, // Start inactive until tested
        ]);

        return redirect()->route('admin.notifications.index')
            ->with('success', 'Provider created successfully. Please test the configuration before activating it.');
    }

    /**
     * Show provider edit form.
     */
    public function editProvider(NotificationProvider $provider)
    {
        $availableProviders = $this->getAvailableProviders($provider->type);
        
        return view('admin.notifications.edit-provider', compact('provider', 'availableProviders'));
    }

    /**
     * Update provider configuration.
     */
    public function updateProvider(Request $request, NotificationProvider $provider)
    {
        $validator = Validator::make($request->all(), [
            'display_name' => 'required|string|max:255',
            'priority' => 'required|integer|min:1|max:10',
            'configuration' => 'required|array',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Validate configuration fields
        $requiredFields = $provider->getRequiredConfigurationFields();
        foreach ($requiredFields as $field) {
            if (!isset($request->configuration[$field]) || empty($request->configuration[$field])) {
                return back()->withErrors(["configuration.{$field}" => "The {$field} field is required"]);
            }
        }

        $provider->update([
            'display_name' => $request->display_name,
            'configuration' => $request->configuration,
            'priority' => $request->priority,
        ]);

        return redirect()->route('admin.notifications.index')
            ->with('success', 'Provider updated successfully. Please test the configuration.');
    }

    /**
     * Test provider configuration.
     */
    public function testProvider(NotificationProvider $provider)
    {
        $success = $this->notificationService->testProvider($provider);

        if ($success) {
            return back()->with('success', "Test successful! The {$provider->display_name} provider is working correctly.");
        } else {
            $errorMessage = $provider->test_results['message'] ?? 'Unknown error occurred';
            return back()->with('error', "Test failed: {$errorMessage}");
        }
    }

    /**
     * Toggle provider active status.
     */
    public function toggleProvider(NotificationProvider $provider)
    {
        $provider->update(['is_active' => !$provider->is_active]);

        $status = $provider->is_active ? 'activated' : 'deactivated';
        return back()->with('success', "Provider {$status} successfully.");
    }

    /**
     * Set provider as default.
     */
    public function setDefault(NotificationProvider $provider)
    {
        if (!$provider->is_active) {
            return back()->with('error', 'Provider must be active to set as default.');
        }

        // Remove default from other providers of the same type
        NotificationProvider::byType($provider->type)->update(['is_default' => false]);

        // Set this provider as default
        $provider->update(['is_default' => true]);

        return back()->with('success', "Provider set as default for {$provider->type} notifications.");
    }

    /**
     * Delete provider.
     */
    public function deleteProvider(NotificationProvider $provider)
    {
        if ($provider->is_default) {
            return back()->with('error', 'Cannot delete the default provider. Set another provider as default first.');
        }

        $provider->delete();

        return back()->with('success', 'Provider deleted successfully.');
    }

    /**
     * Update general notification settings.
     */
    public function updateSettings(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email_enabled' => 'required|boolean',
            'sms_enabled' => 'required|boolean',
            'default_email_from' => 'required|email',
            'test_email' => 'required|email',
            'test_phone' => 'required|string',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        NotificationSetting::set('email_enabled', $request->email_enabled, 'Enable email notifications', 'boolean');
        NotificationSetting::set('sms_enabled', $request->sms_enabled, 'Enable SMS notifications', 'boolean');
        NotificationSetting::set('default_email_from', $request->default_email_from, 'Default email from address');
        NotificationSetting::set('test_email', $request->test_email, 'Test email address');
        NotificationSetting::set('test_phone', $request->test_phone, 'Test phone number');

        return back()->with('success', 'Settings updated successfully.');
    }

    /**
     * Get available providers for a type.
     */
    protected function getAvailableProviders($type)
    {
        if ($type === 'email') {
            return [
                'resend' => [
                    'display_name' => 'Resend',
                    'description' => 'Modern email API with excellent deliverability',
                ],
                'mailgun' => [
                    'display_name' => 'Mailgun',
                    'description' => 'Powerful email service by Rackspace',
                ],
                'sendgrid' => [
                    'display_name' => 'SendGrid',
                    'description' => 'Email delivery service by Twilio',
                ],
            ];
        } else {
            return [
                'termii' => [
                    'display_name' => 'Termii',
                    'description' => 'African SMS and voice communication platform',
                ],
                'twilio' => [
                    'display_name' => 'Twilio',
                    'description' => 'Global SMS and voice API platform',
                ],
                'nexmo' => [
                    'display_name' => 'Nexmo (Vonage)',
                    'description' => 'Global SMS and voice API platform',
                ],
            ];
        }
    }
}
