<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PopupNotification;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class PopupNotificationController extends Controller
{
    /**
     * Display a listing of popup notifications
     */
    public function index(Request $request)
    {
        $query = PopupNotification::with('creator')
                                 ->latest('created_at');

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Filter by status
        if ($request->filled('status')) {
            if ($request->status === 'active') {
                $query->active();
            } elseif ($request->status === 'inactive') {
                $query->where('is_active', false);
            }
        }

        $notifications = $query->paginate(15);

        return view('admin.popup-notifications.index', compact('notifications'));
    }

    /**
     * Show the form for creating a new popup notification
     */
    public function create()
    {
        $levels = range(1, 5); // User levels
        return view('admin.popup-notifications.create', compact('levels'));
    }

    /**
     * Store a newly created popup notification
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => ['required', Rule::in([PopupNotification::TYPE_INFO, PopupNotification::TYPE_WARNING, PopupNotification::TYPE_SUCCESS, PopupNotification::TYPE_ANNOUNCEMENT])],
            'is_active' => 'boolean',
            'start_date' => 'nullable|date|after_or_equal:now',
            'end_date' => 'nullable|date|after:start_date',
            'target_type' => 'required|in:all,levels,users',
            'target_levels' => 'nullable|array',
            'target_levels.*' => 'integer|min:1|max:5',
            'min_level' => 'nullable|integer|min:1|max:5',
            'target_users' => 'nullable|string', // Comma separated usernames
        ]);

        // Process targeting
        $targetUsers = null;
        if ($validated['target_type'] === 'levels') {
            $targetUsers = ['levels' => $validated['target_levels'] ?? []];
            if ($validated['min_level']) {
                $targetUsers['min_level'] = $validated['min_level'];
            }
        } elseif ($validated['target_type'] === 'users' && $validated['target_users']) {
            $usernames = array_map('trim', explode(',', $validated['target_users']));
            $userIds = User::whereIn('username', $usernames)->pluck('id')->toArray();
            $targetUsers = ['user_ids' => $userIds];
        }

        PopupNotification::create([
            'title' => $validated['title'],
            'message' => $validated['message'],
            'type' => $validated['type'],
            'is_active' => $validated['is_active'] ?? true,
            'start_date' => $validated['start_date'],
            'end_date' => $validated['end_date'],
            'target_users' => $targetUsers,
            'created_by' => Auth::id(),
        ]);

        return redirect()->route('admin.popup-notifications.index')
                        ->with('success', 'Popup notification created successfully!');
    }

    /**
     * Display the specified popup notification
     */
    public function show(PopupNotification $popupNotification)
    {
        $popupNotification->load('creator', 'seenBy');
        return view('admin.popup-notifications.show', compact('popupNotification'));
    }

    /**
     * Show the form for editing the specified popup notification
     */
    public function edit(PopupNotification $popupNotification)
    {
        $levels = range(1, 5);
        return view('admin.popup-notifications.edit', compact('popupNotification', 'levels'));
    }

    /**
     * Update the specified popup notification
     */
    public function update(Request $request, PopupNotification $popupNotification)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => ['required', Rule::in([PopupNotification::TYPE_INFO, PopupNotification::TYPE_WARNING, PopupNotification::TYPE_SUCCESS, PopupNotification::TYPE_ANNOUNCEMENT])],
            'is_active' => 'boolean',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
            'target_type' => 'required|in:all,levels,users',
            'target_levels' => 'nullable|array',
            'target_levels.*' => 'integer|min:1|max:5',
            'min_level' => 'nullable|integer|min:1|max:5',
            'target_users' => 'nullable|string',
        ]);

        // Process targeting
        $targetUsers = null;
        if ($validated['target_type'] === 'levels') {
            $targetUsers = ['levels' => $validated['target_levels'] ?? []];
            if ($validated['min_level']) {
                $targetUsers['min_level'] = $validated['min_level'];
            }
        } elseif ($validated['target_type'] === 'users' && $validated['target_users']) {
            $usernames = array_map('trim', explode(',', $validated['target_users']));
            $userIds = User::whereIn('username', $usernames)->pluck('id')->toArray();
            $targetUsers = ['user_ids' => $userIds];
        }

        $popupNotification->update([
            'title' => $validated['title'],
            'message' => $validated['message'],
            'type' => $validated['type'],
            'is_active' => $validated['is_active'] ?? true,
            'start_date' => $validated['start_date'],
            'end_date' => $validated['end_date'],
            'target_users' => $targetUsers,
        ]);

        return redirect()->route('admin.popup-notifications.index')
                        ->with('success', 'Popup notification updated successfully!');
    }

    /**
     * Remove the specified popup notification
     */
    public function destroy(PopupNotification $popupNotification)
    {
        $popupNotification->delete();

        return redirect()->route('admin.popup-notifications.index')
                        ->with('success', 'Popup notification deleted successfully!');
    }

    /**
     * Toggle notification active status
     */
    public function toggle(PopupNotification $popupNotification)
    {
        $popupNotification->update([
            'is_active' => !$popupNotification->is_active
        ]);

        $status = $popupNotification->is_active ? 'activated' : 'deactivated';
        
        return redirect()->back()
                        ->with('success', "Popup notification {$status} successfully!");
    }
}
