<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Post;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class PostController extends Controller
{
    /**
     * Display a listing of posts
     */
    public function index(Request $request)
    {
        $query = Post::with('author')
                    ->latest('created_at');

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Search by title
        if ($request->filled('search')) {
            $query->where('title', 'like', '%' . $request->search . '%');
        }

        $posts = $query->paginate(15);

        return view('admin.posts.index', compact('posts'));
    }

    /**
     * Show the form for creating a new post
     */
    public function create()
    {
        return view('admin.posts.create');
    }

    /**
     * Store a newly created post
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'excerpt' => 'nullable|string|max:500',
            'type' => ['required', Rule::in([Post::TYPE_BLOG, Post::TYPE_NEWS, Post::TYPE_HIGHLIGHT, Post::TYPE_TOURNAMENT, Post::TYPE_UPDATE])],
            'status' => ['required', Rule::in([Post::STATUS_DRAFT, Post::STATUS_PUBLISHED, Post::STATUS_ARCHIVED])],
            'featured_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'published_at' => 'nullable|date',
            'metadata' => 'nullable|array',
        ]);

        // Generate unique slug
        $validated['slug'] = Post::generateSlug($validated['title']);
        $validated['author_id'] = Auth::id();

        // Handle image upload
        if ($request->hasFile('featured_image')) {
            $path = $request->file('featured_image')->store('posts', 'public');
            $validated['featured_image'] = $path;
        }

        // Set published_at if status is published and no date provided
        if ($validated['status'] === Post::STATUS_PUBLISHED && !$validated['published_at']) {
            $validated['published_at'] = now();
        }

        $post = Post::create($validated);

        return redirect()->route('admin.posts.index')
                        ->with('success', 'Post created successfully!');
    }

    /**
     * Display the specified post
     */
    public function show(Post $post)
    {
        $post->load('author');
        return view('admin.posts.show', compact('post'));
    }

    /**
     * Show the form for editing the specified post
     */
    public function edit(Post $post)
    {
        return view('admin.posts.edit', compact('post'));
    }

    /**
     * Update the specified post
     */
    public function update(Request $request, Post $post)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'excerpt' => 'nullable|string|max:500',
            'type' => ['required', Rule::in([Post::TYPE_BLOG, Post::TYPE_NEWS, Post::TYPE_HIGHLIGHT, Post::TYPE_TOURNAMENT, Post::TYPE_UPDATE])],
            'status' => ['required', Rule::in([Post::STATUS_DRAFT, Post::STATUS_PUBLISHED, Post::STATUS_ARCHIVED])],
            'featured_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'published_at' => 'nullable|date',
            'metadata' => 'nullable|array',
        ]);

        // Update slug if title changed
        if ($validated['title'] !== $post->title) {
            $validated['slug'] = Post::generateSlug($validated['title']);
        }

        // Handle image upload
        if ($request->hasFile('featured_image')) {
            // Delete old image
            if ($post->featured_image) {
                Storage::disk('public')->delete($post->featured_image);
            }
            
            $path = $request->file('featured_image')->store('posts', 'public');
            $validated['featured_image'] = $path;
        }

        // Set published_at if changing to published and no date provided
        if ($validated['status'] === Post::STATUS_PUBLISHED && 
            $post->status !== Post::STATUS_PUBLISHED && 
            !$validated['published_at']) {
            $validated['published_at'] = now();
        }

        $post->update($validated);

        return redirect()->route('admin.posts.index')
                        ->with('success', 'Post updated successfully!');
    }

    /**
     * Remove the specified post
     */
    public function destroy(Post $post)
    {
        // Delete associated image
        if ($post->featured_image) {
            Storage::disk('public')->delete($post->featured_image);
        }

        $post->delete();

        return redirect()->route('admin.posts.index')
                        ->with('success', 'Post deleted successfully!');
    }

    /**
     * Show pending posts for approval
     */
    public function pending()
    {
        $posts = Post::pending()
            ->with('author')
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('admin.posts.pending', compact('posts'));
    }

    /**
     * Approve a pending post
     */
    public function approve(Post $post)
    {
        if (!$post->pending_approval || $post->status !== 'draft') {
            return back()->with('error', 'Only pending posts can be approved.');
        }

        $post->update([
            'status' => Post::STATUS_PUBLISHED,
            'pending_approval' => false,
            'published_at' => now(),
        ]);

        return back()->with('success', "Post '{$post->title}' approved and published!");
    }

    /**
     * Reject a pending post
     */
    public function reject(Post $post)
    {
        if (!$post->pending_approval || $post->status !== 'draft') {
            return back()->with('error', 'Only pending posts can be rejected.');
        }

        $post->update([
            'pending_approval' => false,
        ]);

        return back()->with('success', "Post '{$post->title}' rejected and returned to draft.");
    }
}
