<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PlatformSetting;
use App\Services\PaymentService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;

class SettingsController extends Controller
{
    public function __construct(
        private PaymentService $paymentService
    ) {}
    /**
     * Display the settings dashboard.
     */
    public function index()
    {
        $groups = [
            'general' => 'General Settings',
            'payment' => 'Payment Gateways',
            'api' => 'API Configuration',
            'security' => 'Security Settings',
            'platform' => 'Platform Configuration',
            'features' => 'Feature Settings',
        ];

        $stats = [
            'total_settings' => PlatformSetting::count(),
            'payment_gateways' => PlatformSetting::where('group', 'payment')->count(),
            'security_settings' => PlatformSetting::where('group', 'security')->count(),
            'encrypted_settings' => PlatformSetting::where('is_encrypted', true)->count(),
        ];

        return view('admin.settings.index', compact('groups', 'stats'));
    }

    /**
     * Display settings by group.
     */
    public function group(string $group)
    {
        $validGroups = ['general', 'payment', 'api', 'security', 'platform', 'features'];
        
        if (!in_array($group, $validGroups)) {
            abort(404);
        }

        $settings = PlatformSetting::where('group', $group)->orderBy('key')->get();
        $groupTitle = ucfirst($group) . ' Settings';

        return view('admin.settings.group', compact('group', 'groupTitle', 'settings'));
    }

    /**
     * Update settings for a group.
     */
    public function updateGroup(Request $request, string $group)
    {
        $validGroups = ['general', 'payment', 'api', 'security', 'platform', 'features'];
        
        if (!in_array($group, $validGroups)) {
            abort(404);
        }

        // Define validation rules based on group
        $rules = $this->getValidationRules($group);
        $request->validate($rules);

        // Update settings based on group
        switch ($group) {
            case 'general':
                $this->updateGeneralSettings($request);
                break;
            case 'payment':
                $this->updatePaymentSettings($request);
                break;
            case 'api':
                $this->updateApiSettings($request);
                break;
            case 'security':
                $this->updateSecuritySettings($request);
                break;
            case 'platform':
                $this->updatePlatformSettings($request);
                break;
            case 'features':
                $this->updateFeatureSettings($request);
                break;
        }

        // Clear cache to ensure settings take effect
        Cache::flush();

        return redirect()->route('admin.settings.group', $group)
            ->with('success', ucfirst($group) . ' settings updated successfully!');
    }

    /**
     * Test a specific configuration (e.g., payment gateway).
     */
    public function test(Request $request)
    {
        $type = $request->input('type');
        $provider = $request->input('provider');

        try {
            switch ($type) {
                case 'payment':
                    return $this->testPaymentGateway($provider);
                case 'mail':
                    return $this->testMailConfiguration();
                case 'sms':
                    return $this->testSMSConfiguration($provider);
                case 'api':
                    return $this->testApiConfiguration($provider);
                default:
                    return response()->json(['success' => false, 'message' => 'Invalid test type']);
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    /**
     * Clear all application caches.
     */
    public function clearCache()
    {
        try {
            Artisan::call('cache:clear');
            Artisan::call('config:clear');
            Artisan::call('view:clear');
            Artisan::call('route:clear');

            return redirect()->back()->with('success', 'All caches cleared successfully!');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to clear caches: ' . $e->getMessage());
        }
    }

    /**
     * Get validation rules for each group.
     */
    private function getValidationRules(string $group): array
    {
        return match ($group) {
            'general' => [
                'platform_name' => 'nullable|string|max:255',
                'platform_description' => 'nullable|string',
                'platform_logo' => 'nullable|url',
                'support_email' => 'nullable|email',
                'contact_phone' => 'nullable|string|max:20',
                'timezone' => 'nullable|string|max:50',
                'currency' => 'nullable|string|max:10',
            ],
            'payment' => [
                'paystack_public_key' => 'nullable|string',
                'paystack_secret_key' => 'nullable|string',
                'flutterwave_public_key' => 'nullable|string',
                'flutterwave_secret_key' => 'nullable|string',
                'paypal_client_id' => 'nullable|string',
                'paypal_secret' => 'nullable|string',
                'stripe_publishable_key' => 'nullable|string',
                'stripe_secret_key' => 'nullable|string',
            ],
            'api' => [
                'qoreid_api_key' => 'nullable|string',
                'qoreid_secret_key' => 'nullable|string',
                'qoreid_base_url' => 'nullable|url',
                'twilio_account_sid' => 'nullable|string',
                'twilio_auth_token' => 'nullable|string',
                'twilio_phone_number' => 'nullable|string',
                'firebase_server_key' => 'nullable|string',
                'pusher_app_id' => 'nullable|string',
                'pusher_key' => 'nullable|string',
                'pusher_secret' => 'nullable|string',
                'pusher_cluster' => 'nullable|string',
            ],
            'security' => [
                'max_login_attempts' => 'nullable|integer|min:1|max:20',
                'session_lifetime' => 'nullable|integer|min:60|max:43200',
                'two_factor_enabled' => 'nullable|boolean',
                'ip_whitelist_enabled' => 'nullable|boolean',
                'rate_limit_per_minute' => 'nullable|integer|min:10|max:1000',
            ],
            'platform' => [
                'commission_rate' => 'nullable|numeric|min:0|max:100',
                'minimum_deposit' => 'nullable|numeric|min:1',
                'maximum_deposit' => 'nullable|numeric|min:1',
                'minimum_withdrawal' => 'nullable|numeric|min:1',
                'maximum_withdrawal' => 'nullable|numeric|min:1',
                'kyc_required_for_withdrawal' => 'nullable|boolean',
            ],
            'features' => [
                'challenges_enabled' => 'nullable|boolean',
                'tournaments_enabled' => 'nullable|boolean',
                'wallet_enabled' => 'nullable|boolean',
                'referral_enabled' => 'nullable|boolean',
                'level_system_enabled' => 'nullable|boolean',
            ],
            default => [],
        };
    }

    /**
     * Update general settings.
     */
    private function updateGeneralSettings(Request $request): void
    {
        $settings = [
            'platform_name' => ['value' => $request->platform_name, 'description' => 'Platform name displayed across the site'],
            'platform_description' => ['value' => $request->platform_description, 'description' => 'Brief description of the platform'],
            'platform_logo' => ['value' => $request->platform_logo, 'description' => 'Platform logo URL or path'],
            'support_email' => ['value' => $request->support_email, 'description' => 'Support email address'],
            'contact_phone' => ['value' => $request->contact_phone, 'description' => 'Contact phone number'],
            'timezone' => ['value' => $request->timezone, 'description' => 'Default timezone for the platform'],
            'currency' => ['value' => $request->currency, 'description' => 'Default currency symbol'],
        ];

        foreach ($settings as $key => $data) {
            if ($data['value'] !== null) {
                PlatformSetting::set($key, $data['value'], 'string', false, 'general', $data['description']);
            }
        }
    }

    /**
     * Update payment settings.
     */
    private function updatePaymentSettings(Request $request): void
    {
        $settings = [
            'paystack_public_key' => ['value' => $request->paystack_public_key, 'description' => 'Paystack public key'],
            'paystack_secret_key' => ['value' => $request->paystack_secret_key, 'description' => 'Paystack secret key', 'encrypt' => true],
            'flutterwave_public_key' => ['value' => $request->flutterwave_public_key, 'description' => 'Flutterwave public key'],
            'flutterwave_secret_key' => ['value' => $request->flutterwave_secret_key, 'description' => 'Flutterwave secret key', 'encrypt' => true],
            'paypal_client_id' => ['value' => $request->paypal_client_id, 'description' => 'PayPal client ID'],
            'paypal_secret' => ['value' => $request->paypal_secret, 'description' => 'PayPal secret', 'encrypt' => true],
            'stripe_publishable_key' => ['value' => $request->stripe_publishable_key, 'description' => 'Stripe publishable key'],
            'stripe_secret_key' => ['value' => $request->stripe_secret_key, 'description' => 'Stripe secret key', 'encrypt' => true],
        ];

        foreach ($settings as $key => $data) {
            if ($data['value'] !== null) {
                PlatformSetting::set(
                    $key, 
                    $data['value'], 
                    'string', 
                    $data['encrypt'] ?? false, 
                    'payment', 
                    $data['description']
                );
            }
        }
    }

    /**
     * Update API settings.
     */
    private function updateApiSettings(Request $request): void
    {
        $settings = [
            'qoreid_api_key' => ['value' => $request->qoreid_api_key, 'description' => 'QoreID API key for KYC verification', 'encrypt' => true],
            'qoreid_secret_key' => ['value' => $request->qoreid_secret_key, 'description' => 'QoreID secret key for webhook verification', 'encrypt' => true],
            'qoreid_base_url' => ['value' => $request->qoreid_base_url, 'description' => 'QoreID API base URL'],
            'twilio_account_sid' => ['value' => $request->twilio_account_sid, 'description' => 'Twilio Account SID for SMS'],
            'twilio_auth_token' => ['value' => $request->twilio_auth_token, 'description' => 'Twilio Auth Token', 'encrypt' => true],
            'twilio_phone_number' => ['value' => $request->twilio_phone_number, 'description' => 'Twilio phone number'],
            'firebase_server_key' => ['value' => $request->firebase_server_key, 'description' => 'Firebase server key for push notifications', 'encrypt' => true],
            'pusher_app_id' => ['value' => $request->pusher_app_id, 'description' => 'Pusher App ID for real-time updates'],
            'pusher_key' => ['value' => $request->pusher_key, 'description' => 'Pusher Key'],
            'pusher_secret' => ['value' => $request->pusher_secret, 'description' => 'Pusher Secret', 'encrypt' => true],
            'pusher_cluster' => ['value' => $request->pusher_cluster, 'description' => 'Pusher Cluster'],
        ];

        foreach ($settings as $key => $data) {
            if ($data['value'] !== null) {
                PlatformSetting::set(
                    $key, 
                    $data['value'], 
                    'string', 
                    $data['encrypt'] ?? false, 
                    'api', 
                    $data['description']
                );
            }
        }
    }

    /**
     * Update security settings.
     */
    private function updateSecuritySettings(Request $request): void
    {
        $settings = [
            'max_login_attempts' => ['value' => $request->max_login_attempts, 'type' => 'integer', 'description' => 'Maximum login attempts before lockout'],
            'session_lifetime' => ['value' => $request->session_lifetime, 'type' => 'integer', 'description' => 'Session lifetime in minutes'],
            'two_factor_enabled' => ['value' => $request->boolean('two_factor_enabled'), 'type' => 'boolean', 'description' => 'Enable two-factor authentication'],
            'ip_whitelist_enabled' => ['value' => $request->boolean('ip_whitelist_enabled'), 'type' => 'boolean', 'description' => 'Enable IP whitelisting for admin'],
            'rate_limit_per_minute' => ['value' => $request->rate_limit_per_minute, 'type' => 'integer', 'description' => 'Rate limit per minute for API requests'],
        ];

        foreach ($settings as $key => $data) {
            if ($data['value'] !== null) {
                PlatformSetting::set($key, $data['value'], $data['type'], false, 'security', $data['description']);
            }
        }
    }

    /**
     * Update platform settings.
     */
    private function updatePlatformSettings(Request $request): void
    {
        $settings = [
            'commission_rate' => ['value' => $request->commission_rate, 'type' => 'float', 'description' => 'Platform commission rate (percentage)'],
            'minimum_deposit' => ['value' => $request->minimum_deposit, 'type' => 'float', 'description' => 'Minimum deposit amount'],
            'maximum_deposit' => ['value' => $request->maximum_deposit, 'type' => 'float', 'description' => 'Maximum deposit amount'],
            'minimum_withdrawal' => ['value' => $request->minimum_withdrawal, 'type' => 'float', 'description' => 'Minimum withdrawal amount'],
            'maximum_withdrawal' => ['value' => $request->maximum_withdrawal, 'type' => 'float', 'description' => 'Maximum withdrawal amount'],
            'kyc_required_for_withdrawal' => ['value' => $request->boolean('kyc_required_for_withdrawal'), 'type' => 'boolean', 'description' => 'Require KYC for withdrawals'],
        ];

        foreach ($settings as $key => $data) {
            if ($data['value'] !== null) {
                PlatformSetting::set($key, $data['value'], $data['type'], false, 'platform', $data['description']);
            }
        }
    }

    /**
     * Update feature settings.
     */
    private function updateFeatureSettings(Request $request): void
    {
        $settings = [
            'challenges_enabled' => ['value' => $request->boolean('challenges_enabled'), 'type' => 'boolean', 'description' => 'Enable challenge system'],
            'tournaments_enabled' => ['value' => $request->boolean('tournaments_enabled'), 'type' => 'boolean', 'description' => 'Enable tournament system'],
            'wallet_enabled' => ['value' => $request->boolean('wallet_enabled'), 'type' => 'boolean', 'description' => 'Enable wallet system'],
            'referral_enabled' => ['value' => $request->boolean('referral_enabled'), 'type' => 'boolean', 'description' => 'Enable referral system'],
            'level_system_enabled' => ['value' => $request->boolean('level_system_enabled'), 'type' => 'boolean', 'description' => 'Enable level progression system'],
        ];

        foreach ($settings as $key => $data) {
            PlatformSetting::set($key, $data['value'], $data['type'], false, 'features', $data['description']);
        }
    }

    private function testPaymentGateway(string $provider): array
    {
        try {
            return $this->paymentService->testProvider($provider);
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Test mail configuration.
     */
    private function testMailConfiguration(): array
    {
        // This would contain actual mail testing logic
        return ['success' => true, 'message' => 'Mail configuration test completed successfully'];
    }

    /**
     * Test SMS configuration.
     */
    private function testSMSConfiguration(string $provider): array
    {
        // This would contain actual SMS testing logic
        return ['success' => true, 'message' => ucfirst($provider) . ' SMS configuration test completed successfully'];
    }

    /**
     * Test API configuration.
     */
    private function testApiConfiguration(string $provider): array
    {
        try {
            switch ($provider) {
                case 'qoreid':
                    $qoreIdService = app(\App\Services\QoreIdService::class);
                    return $qoreIdService->testConnection();
                    
                case 'twilio':
                    // Test Twilio connection
                    return ['success' => true, 'message' => 'Twilio API connection test completed successfully'];
                    
                case 'firebase':
                    // Test Firebase connection
                    return ['success' => true, 'message' => 'Firebase API connection test completed successfully'];
                    
                case 'pusher':
                    // Test Pusher connection
                    return ['success' => true, 'message' => 'Pusher API connection test completed successfully'];
                    
                default:
                    return ['success' => false, 'message' => 'Unknown API provider'];
            }
        } catch (\Exception $e) {
            return ['success' => false, 'message' => 'API test failed: ' . $e->getMessage()];
        }
    }
}
