<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Tournament;
use App\Models\TournamentParticipant;
use App\Models\Game;
use App\Models\User;
use App\Services\WalletService;
use Illuminate\Http\Request;

class TournamentController extends Controller
{
    protected $walletService;

    public function __construct(WalletService $walletService)
    {
        $this->walletService = $walletService;
    }

    public function index()
    {
        $tournaments = Tournament::with(['host', 'game'])
                                ->withCount('participants')
                                ->orderBy('created_at', 'desc')
                                ->paginate(15);

        return view('admin.tournaments.index', compact('tournaments'));
    }

    public function create()
    {
        $games = Game::orderBy('name')->get();
        $hosts = User::where('is_organizer', true)->orderBy('username')->get();

        return view('admin.tournaments.create', compact('games', 'hosts'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'host_id' => 'required|exists:users,id',
            'game_id' => 'required|exists:games,id',
            'type' => 'required|in:single_elimination,double_elimination,round_robin',
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'entry_fee' => 'required|numeric|min:0',
            'prize_pool' => 'required|numeric|min:0',
            'livestream_link' => 'nullable|url',
            'starts_at' => 'required|date|after:now',
            'ends_at' => 'required|date|after:starts_at',
        ]);

        Tournament::create($request->all() + ['status' => 'upcoming']);

        return redirect()->route('admin.tournaments.index')
                        ->with('success', 'Tournament created successfully!');
    }

    public function show(Tournament $tournament)
    {
        $tournament->load(['host', 'game', 'participants.user', 'matches']);
        
        return view('admin.tournaments.show', compact('tournament'));
    }

    public function edit(Tournament $tournament)
    {
        $games = Game::orderBy('name')->get();
        $hosts = User::where('is_organizer', true)->orderBy('username')->get();

        return view('admin.tournaments.edit', compact('tournament', 'games', 'hosts'));
    }

    public function update(Request $request, Tournament $tournament)
    {
        $request->validate([
            'host_id' => 'required|exists:users,id',
            'game_id' => 'required|exists:games,id',
            'type' => 'required|in:single_elimination,double_elimination,round_robin',
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'entry_fee' => 'required|numeric|min:0',
            'prize_pool' => 'required|numeric|min:0',
            'livestream_link' => 'nullable|url',
            'starts_at' => 'required|date',
            'ends_at' => 'required|date|after:starts_at',
            'status' => 'required|in:upcoming,ongoing,completed,cancelled',
        ]);

        $tournament->update($request->all());

        return redirect()->route('admin.tournaments.index')
                        ->with('success', 'Tournament updated successfully!');
    }

    public function destroy(Tournament $tournament)
    {
        if ($tournament->participants()->count() > 0) {
            return redirect()->route('admin.tournaments.index')
                            ->with('error', 'Cannot delete tournament with participants.');
        }

        $tournament->delete();

        return redirect()->route('admin.tournaments.index')
                        ->with('success', 'Tournament deleted successfully!');
    }

    public function participants(Tournament $tournament)
    {
        $participants = $tournament->participants()->with('user')->get();
        $availableUsers = User::whereNotIn('id', $participants->pluck('user_id'))
                             ->where('wallet_balance', '>=', $tournament->entry_fee)
                             ->orderBy('username')
                             ->get();

        return view('admin.tournaments.participants', compact('tournament', 'participants', 'availableUsers'));
    }

    public function addParticipant(Request $request, Tournament $tournament)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
        ]);

        $user = User::findOrFail($request->user_id);

        // Check if user is already participant
        if ($tournament->participants()->where('user_id', $user->id)->exists()) {
            return redirect()->back()->with('error', 'User is already a participant.');
        }

        // Check if user has enough balance
        if ($user->wallet_balance < $tournament->entry_fee) {
            return redirect()->back()->with('error', 'User does not have sufficient balance.');
        }

        // Deduct entry fee and add participant
        $this->walletService->deductBalance($user, $tournament->entry_fee, 'tournament_entry', 
                                           "Tournament entry fee for: {$tournament->title}");

        TournamentParticipant::create([
            'tournament_id' => $tournament->id,
            'user_id' => $user->id,
            'seed' => $tournament->participants()->count() + 1,
        ]);

        return redirect()->back()->with('success', 'Participant added successfully!');
    }

    public function removeParticipant(Tournament $tournament, TournamentParticipant $participant)
    {
        if ($tournament->status !== 'upcoming') {
            return redirect()->back()->with('error', 'Cannot remove participants from started tournaments.');
        }

        // Refund entry fee
        $this->walletService->addBalance($participant->user, $tournament->entry_fee, 'tournament_refund',
                                        "Refund for tournament: {$tournament->title}");

        $participant->delete();

        return redirect()->back()->with('success', 'Participant removed and refunded successfully!');
    }

    public function startTournament(Tournament $tournament)
    {
        if ($tournament->status !== 'upcoming') {
            return redirect()->back()->with('error', 'Tournament has already started or ended.');
        }

        if ($tournament->participants()->count() < 2) {
            return redirect()->back()->with('error', 'Tournament needs at least 2 participants to start.');
        }

        $tournament->update(['status' => 'ongoing']);

        return redirect()->back()->with('success', 'Tournament started successfully!');
    }

    public function completeTournament(Tournament $tournament)
    {
        if ($tournament->status !== 'ongoing') {
            return redirect()->back()->with('error', 'Tournament is not currently ongoing.');
        }

        $tournament->update(['status' => 'completed']);

        return redirect()->back()->with('success', 'Tournament completed successfully!');
    }
}