<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Level;
use App\Models\User;
use App\Models\WalletTransaction;
use App\Services\LevelService;
use App\Services\RoleManagementService;
use App\Enums\UserRole;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class UserController extends Controller
{
    protected LevelService $levelService;
    protected RoleManagementService $roleService;

    public function __construct(LevelService $levelService, RoleManagementService $roleService)
    {
        $this->levelService = $levelService;
        $this->roleService = $roleService;
    }

    /**
     * Display a listing of users.
     */
    public function index(Request $request)
    {
        $query = User::query();

        if ($request->has('search') && $request->get('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('username', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->has('status') && $request->get('status')) {
            $query->where('status', $request->get('status'));
        }

        if ($request->has('level') && $request->get('level')) {
            $query->where('level', $request->get('level'));
        }

        if ($request->has('role') && $request->get('role')) {
            $query->where('role', $request->get('role'));
        }

        if ($request->has('kyc_status') && $request->get('kyc_status')) {
            $query->where('kyc_status', $request->get('kyc_status'));
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortDirection = $request->get('sort_direction', 'desc');

        if (in_array($sortBy, ['username', 'email', 'level', 'wallet_balance', 'xp', 'created_at'])) {
            $query->orderBy($sortBy, $sortDirection);
        } else {
            $query->latest();
        }

        $users = $query->paginate(20);
        $levels = Level::all();

        return view('admin.users.index', compact('users', 'levels'));
    }

    /**
     * Display the specified user.
     */
    public function show(User $user)
    {
        $user->load([
            'createdChallenges',
            'acceptedChallenges',
            'walletTransactions' => function ($query) {
                $query->latest()->limit(20);
            },
            'badges',
        ]);

        $levels = Level::all();
        $processedLevels = $this->levelService->processLevelsForDropdown($levels);
        $recentActivity = $this->getUserActivity($user);
        $availableRoles = UserRole::forDropdown();

        return view('admin.users.show', compact('user', 'levels', 'processedLevels', 'recentActivity', 'availableRoles'));
    }

    /**
     * Ban a user.
     */
    public function ban(User $user)
    {
        $user->update(['status' => 'banned']);

        return back()->with('success', 'User has been banned successfully.');
    }

    /**
     * Unban a user.
     */
    public function unban(User $user)
    {
        $user->update(['status' => 'active']);

        return back()->with('success', 'User has been unbanned successfully.');
    }

    /**
     * Update user's organizer status.
     */
    public function toggleOrganizer(User $user)
    {
        $user->update(['is_organizer' => ! $user->is_organizer]);

        $status = $user->is_organizer ? 'granted' : 'revoked';

        return back()->with('success', "Organizer privileges {$status} successfully.");
    }

    /**
     * Update user's role using new role system.
     */
    public function updateRole(Request $request, User $user)
    {
        $availableRoles = UserRole::values();
        
        $request->validate([
            'role' => 'required|string|in:' . implode(',', $availableRoles),
        ]);

        $oldRole = $user->role;
        $newRole = UserRole::from($request->role);
        
        // Use role management service to handle role change and level adjustment
        $this->roleService->changeUserRole($user, $newRole);

        return back()->with('success', "User role updated from {$oldRole->getDisplayName()} to {$newRole->getDisplayName()} successfully.");
    }

    /**
     * Update user's KYC status.
     */
    public function updateKycStatus(Request $request, User $user)
    {
        $request->validate([
            'kyc_status' => 'required|string|in:none,pending,verified,rejected',
        ]);

        $user->update(['kyc_status' => $request->kyc_status]);

        return back()->with('success', "KYC status updated to {$request->kyc_status} successfully.");
    }

    /**
     * Assign a specific level to user (1-30 or Creator).
     */
    public function assignLevel(Request $request, User $user)
    {
        $request->validate([
            'level_id' => 'required|exists:levels,id',
        ]);

        $level = Level::findOrFail($request->level_id);
        $oldLevel = $user->level;

        $user->update(['level' => $level->id]);

        // If assigning Creator level, also update role to Creator
        if ($level->name === 'Creator') {
            $this->roleService->changeUserRole($user, UserRole::CREATOR);
        } elseif ($level->id >= 29 && $user->role === UserRole::BASIC_USER) {
            // Auto-promote to Premium User if reaching level 29+
            $this->roleService->changeUserRole($user, UserRole::PREMIUM_USER);
        }

        // Log the level assignment
        $this->logUserActivity($user, 'level_assigned', [
            'old_level' => $oldLevel,
            'new_level' => $level->id,
            'level_name' => $level->name,
            'assigned_by' => Auth::user()->username,
        ]);

        return back()->with('success', "User level assigned to {$level->name} successfully.");
    }

    /**
     * Credit user's wallet balance.
     */
    public function creditWallet(Request $request, User $user)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1|max:1000000',
            'description' => 'required|string|max:255',
        ]);

        $amount = $request->amount;
        $description = $request->description;

        DB::transaction(function () use ($user, $amount, $description) {
            // Update user balance
            $user->increment('wallet_balance', $amount);

            // Create transaction record
            WalletTransaction::create([
                'user_id' => $user->id,
                'type' => 'admin_credit',
                'amount' => $amount,
                'status' => 'completed',
                'description' => "Admin Credit: {$description}",
                'metadata' => [
                    'admin_id' => Auth::id(),
                    'admin_username' => Auth::user()->username,
                    'original_description' => $description,
                ],
            ]);

            // Log the activity
            $this->logUserActivity($user, 'wallet_credited', [
                'amount' => $amount,
                'description' => $description,
                'credited_by' => Auth::user()->username,
                'new_balance' => $user->fresh()->wallet_balance,
            ]);
        });

        return back()->with('success', '₦'.number_format($amount)." credited to user's wallet successfully.");
    }

    /**
     * Debit user's wallet balance.
     */
    public function debitWallet(Request $request, User $user)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1|max:'.$user->wallet_balance,
            'description' => 'required|string|max:255',
        ]);

        $amount = $request->amount;
        $description = $request->description;

        DB::transaction(function () use ($user, $amount, $description) {
            // Update user balance
            $user->decrement('wallet_balance', $amount);

            // Create transaction record
            WalletTransaction::create([
                'user_id' => $user->id,
                'type' => 'admin_debit',
                'amount' => -$amount,
                'status' => 'completed',
                'description' => "Admin Debit: {$description}",
                'metadata' => [
                    'admin_id' => Auth::id(),
                    'admin_username' => Auth::user()->username,
                    'original_description' => $description,
                ],
            ]);

            // Log the activity
            $this->logUserActivity($user, 'wallet_debited', [
                'amount' => $amount,
                'description' => $description,
                'debited_by' => Auth::user()->username,
                'new_balance' => $user->fresh()->wallet_balance,
            ]);
        });

        return back()->with('success', '₦'.number_format($amount)." debited from user's wallet successfully.");
    }

    /**
     * Get comprehensive user activity data.
     */
    private function getUserActivity(User $user)
    {
        $activities = collect();

        // Recent challenges
        $challenges = $user->createdChallenges()
            ->orWhere('accepter_id', $user->id)
            ->latest()
            ->limit(10)
            ->get()
            ->map(function ($challenge) use ($user) {
                return [
                    'type' => 'challenge',
                    'action' => $challenge->creator_id === $user->id ? 'created' : 'accepted',
                    'description' => "Challenge: {$challenge->title}",
                    'amount' => $challenge->wager_amount,
                    'created_at' => $challenge->created_at,
                    'details' => $challenge,
                ];
            });

        // Wallet transactions
        $transactions = $user->walletTransactions()
            ->latest()
            ->limit(15)
            ->get()
            ->map(function ($transaction) {
                return [
                    'type' => 'wallet',
                    'action' => $transaction->type,
                    'description' => $transaction->description,
                    'amount' => $transaction->amount,
                    'created_at' => $transaction->created_at,
                    'details' => $transaction,
                ];
            });

        // Merge and sort all activities
        $activities = $challenges->concat($transactions)
            ->sortByDesc('created_at')
            ->take(20);

        return $activities;
    }

    /**
     * Log user activity for audit purposes.
     */
    private function logUserActivity(User $user, string $action, array $metadata = [])
    {
        // For now, we'll create a wallet transaction entry to track admin actions
        // In a production system, you might want a dedicated admin_actions table
        WalletTransaction::create([
            'user_id' => $user->id,
            'type' => 'admin_action',
            'amount' => 0,
            'status' => 'completed',
            'description' => "Admin Action: {$action}",
            'metadata' => array_merge([
                'action' => $action,
                'admin_id' => Auth::id(),
                'admin_username' => Auth::user()->username,
                'timestamp' => now()->toISOString(),
            ], $metadata),
        ]);
    }
}
