<?php

namespace App\Http\Controllers;

use App\Models\Challenge;
use App\Models\Game;
use App\Notifications\ChallengeInvitation;
use App\Services\WalletService;
use App\Services\XPService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log; // Added: Import Laravel's Log facade
use Illuminate\Support\Facades\Storage;

class ChallengeController extends Controller
{
    public function __construct(
        private WalletService $walletService,
        private XPService $xpService
    ) {}

    /**
     * Display a listing of challenges.
     */
    public function index(Request $request)
    {
        $filter = $request->get('filter', 'global'); // 'global' or 'friends'
        $gameId = $request->get('game_id');
        $wagerRange = $request->get('wager_range');
        $bestOf = $request->get('best_of');

        $query = Challenge::with(['creator', 'accepter', 'game'])
            ->where('status', 'open');

        // Apply friend filter if authenticated and requested
        if (auth()->check() && $filter === 'friends') {
            $friendIds = auth()->user()->friends()->pluck('users.id');
            $query->whereIn('creator_id', $friendIds);
        }

        // Apply game filter
        if ($gameId) {
            $query->where('game_id', $gameId);
        }

        // Apply wager range filter
        if ($wagerRange) {
            switch ($wagerRange) {
                case '100-500':
                    $query->whereBetween('wager_amount', [100, 500]);
                    break;
                case '500-1000':
                    $query->whereBetween('wager_amount', [500, 1000]);
                    break;
                case '1000+':
                    $query->where('wager_amount', '>=', 1000);
                    break;
            }
        }

        // Apply best of filter
        if ($bestOf) {
            $query->where('best_of', $bestOf);
        }

        $challenges = $query->latest()->paginate(15);
        $games = Game::all();

        return view('challenges.index', compact('challenges', 'games', 'filter'));
    }

    /**
     * Display the user's challenges (accepted/played).
     */
    public function myChallenges(Request $request)
    {
        $user = auth()->user();
        $filter = $request->get('filter', 'all'); // 'all', 'ongoing', 'past'
        $gameId = $request->get('game_id');

        $query = Challenge::with(['creator', 'accepter', 'game'])
            ->where(function($q) use ($user) {
                $q->where('creator_id', $user->id)
                  ->orWhere('accepter_id', $user->id);
            });

        // Apply status filter
        switch ($filter) {
            case 'ongoing':
                $query->whereIn('status', ['accepted', 'open']);
                break;
            case 'past':
                $query->where('status', 'completed');
                break;
            default:
                // Show all (ongoing and past)
                $query->whereIn('status', ['open', 'accepted', 'completed']);
                break;
        }

        // Apply game filter
        if ($gameId) {
            $query->where('game_id', $gameId);
        }

        $challenges = $query->latest()->paginate(15);
        $games = Game::all();

        return view('challenges.my', compact('challenges', 'games', 'filter'));
    }

    /**
     * Show the form for creating a new challenge.
     */
    public function create()
    {
        $this->authorize('create', Challenge::class);
        
        $games = Game::all();
        return view('challenges.create', compact('games'));
    }

    /**
     * Store a newly created challenge in storage.
     */
    public function store(Request $request)
    {
        $this->authorize('create', Challenge::class);

        $user = Auth::user();
        
        $request->validate([
            'game_id' => 'required|exists:games,id',
            'wager_amount' => 'required|numeric|min:1|max:' . $user->wallet_balance,
            'best_of' => 'required|integer|min:1|max:7',
            'rules_text' => 'nullable|string|max:1000',
            'creator_game_username' => 'required|string|max:255',
            'group_link' => 'nullable|url|max:500',
            'group_id' => 'nullable|string|max:255',
        ]);

        // Check role-based wager amount limits
        if (!$user->canCreateChallengeWithAmount($request->wager_amount)) {
            $maxAmount = $user->getMaxChallengeAmount();
            $maxAmountFormatted = $maxAmount ? number_format($maxAmount) : 'unlimited';
            return back()->withErrors([
                'wager_amount' => "Your role ({$user->role}) allows challenges up to ₦{$maxAmountFormatted} only."
            ])->withInput();
        }

        $challenge = Challenge::create([
            'creator_id' => Auth::id(),
            'game_id' => $request->game_id,
            'wager_amount' => $request->wager_amount,
            'best_of' => $request->best_of,
            'rules_text' => $request->rules_text,
            'creator_game_username' => $request->creator_game_username,
            'group_link' => $request->group_link,
            'group_id' => $request->group_id,
        ]);

        // Lock the creator's wager amount
        $this->walletService->lockFunds($user, $request->wager_amount, 'challenge', $challenge->id);

        return redirect()->route('challenges.show', $challenge)
            ->with('success', 'Challenge created successfully!');
    }

    /**
     * Display the specified challenge.
     */
    public function show(Challenge $challenge)
    {
        $challenge->load(['creator', 'accepter', 'game', 'matchProofs.user', 'dispute']);
        
        return view('challenges.show', compact('challenge'));
    }

    /**
     * Accept a challenge.
     */
    public function accept(Challenge $challenge)
    {
        $user = Auth::user();

        if (!$challenge->canBeAcceptedBy($user)) {
            return back()->with('error', 'You cannot accept this challenge.');
        }

        // Lock accepter's wager amount
        $this->walletService->lockFunds($user, $challenge->wager_amount, 'challenge', $challenge->id);

        $challenge->update([
            'accepter_id' => $user->id,
            'status' => 'accepted',
        ]);

        // Award XP for accepting challenge
        $this->xpService->awardXP($user, 10, 'challenge_accepted');

        return redirect()->route('challenges.show', $challenge)
            ->with('success', 'Challenge accepted! Upload your proof when ready.');
    }

    /**
     * Submit proof for a challenge.
     */
    public function submitProof(Request $request, Challenge $challenge)
    {
        $user = Auth::user();

        if (!in_array($user->id, [$challenge->creator_id, $challenge->accepter_id])) {
            return back()->with('error', 'You are not part of this challenge.');
        }

        if ($challenge->status !== 'accepted') {
            return back()->with('error', 'Challenge is not in accepted state.');
        }

        $request->validate([
            'proof_file' => 'required|image|max:2048',
            'notes' => 'nullable|string|max:500',
        ]);

        try {
            $filePath = $request->file('proof_file')->store('match_proofs', 'public');

            $challenge->matchProofs()->create([
                'user_id' => $user->id,
                'file_url' => Storage::url($filePath),
                'notes' => $request->notes,
            ]);

            return redirect()->route('challenges.show', $challenge)
                ->with('success', 'Proof submitted successfully!');
        } catch (\Exception $e) {
        } catch (FileNotFoundException | FileException $e) {
            Log::error('File upload error in submitProof: ' . $e->getMessage());
            return back()->with('error', 'Failed to upload proof. Please try again.');
        }
    }

    /**
     * Agree to challenge result.
     */
    /**
     * Handle post-game result claims (I Won / I Lost buttons)
     */
    public function claimResult(Request $request, Challenge $challenge)
    {
        $user = Auth::user();

        // Verify user is part of this challenge
        if (!in_array($user->id, [$challenge->creator_id, $challenge->accepter_id])) {
            return back()->with('error', 'You are not part of this challenge.');
        }

        // Verify challenge is in accepted state
        if ($challenge->status !== 'accepted') {
            return back()->with('error', 'Cannot claim result for this challenge.');
        }

        $request->validate([
            'result' => 'required|in:win,loss',
        ]);

        try {
            // Determine which field to update based on user
            $field = $user->id === $challenge->creator_id ? 'creator_claimed_result' : 'accepter_claimed_result';
            
            // Check if user has already claimed a result
            if ($challenge->$field !== 'none') {
                return back()->with('error', 'You have already claimed a result for this challenge.');
            }
            
            $challenge->update([
                $field => $request->result
            ]);

            // Immediate resolution if current player claims loss
            if ($request->result === 'loss') {
                $winner = $challenge->getWinnerFromLossClaimant();
                $result = $winner->id === $challenge->creator_id ? 'creator_win' : 'accepter_win';
                
                $challenge->update([
                    'status' => 'completed',
                    'result' => $result,
                ]);

                // Process payouts
                $this->walletService->processChallengeWin($challenge);

                // Award XP
                $this->xpService->awardXP($winner, 20, 'challenge_won');

                return redirect()->route('challenges.show', $challenge)
                    ->with('success', 'Match resolved! You forfeited and your opponent wins. Funds have been distributed.');
            }

            // Check if both players have submitted claims (only for non-loss submissions)
            if ($challenge->hasBothClaimedResults()) {
                if ($challenge->resultsMatch()) {
                    // Results match - automatic payout
                    $winner = $challenge->getWinnerFromClaims();
                    $result = $winner->id === $challenge->creator_id ? 'creator_win' : 'accepter_win';
                    
                    $challenge->update([
                        'status' => 'completed',
                        'result' => $result,
                    ]);

                    // Process payouts
                    $this->walletService->processChallengeWin($challenge);

                    // Award XP
                    $this->xpService->awardXP($winner, 20, 'challenge_won');

                    return redirect()->route('challenges.show', $challenge)
                        ->with('success', 'Results confirmed! Funds have been distributed to the winner.');
                } elseif ($challenge->bothClaimVictory()) {
                    // Both claim victory - create dispute requiring evidence
                    $dispute = $challenge->dispute()->create([
                        'opened_by' => $user->id,
                        'reason' => 'Both players claim victory - evidence required for resolution.',
                    ]);

                    $challenge->update(['status' => 'disputed']);

                    return redirect()->route('challenges.show', $challenge)
                        ->with('info', 'Both players claim victory. Please upload evidence to resolve the dispute.');
                } elseif ($challenge->bothClaimLoss()) {
                    // Both claim loss - create dispute for admin review (rare forfeit case)
                    $dispute = $challenge->dispute()->create([
                        'opened_by' => $user->id,
                        'reason' => 'Both players claim loss - mutual forfeit requires admin review.',
                    ]);

                    $challenge->update(['status' => 'disputed']);

                    return redirect()->route('challenges.show', $challenge)
                        ->with('info', 'Both players forfeited. This rare case requires admin review to determine the outcome.');
                } else {
                    // Invalid combination - reset for clarity
                    $challenge->resetResultClaims();
                    return back()->with('error', 'Invalid result combination. Please try again.');
                }
            }

            return back()->with('success', 'Your result has been recorded. Waiting for your opponent.');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return back()
                ->withErrors($e->validator)
                ->withInput();
        } catch (\Exception $e) {
            // Using Laravel's Log facade for proper logging
            Log::error('Error processing result claim: ' . $e->getMessage());
            $errorId = uniqid('err_', true);
            Log::error("Error ID {$errorId}: Error processing result claim", [
                'exception' => $e,
            ]);
            return back()->with('error', "An error occurred while processing your result. Please try again. (Error ID: {$errorId})");
        }
    }

    /**
     * Upload evidence for a dispute when both claim victory
     */
    public function uploadEvidence(Request $request, Challenge $challenge)
    {
        $user = Auth::user();

        // Verify user is part of this challenge
        if (!in_array($user->id, [$challenge->creator_id, $challenge->accepter_id])) {
            return back()->with('error', 'You are not part of this challenge.');
        }

        // Verify challenge is disputed
        if ($challenge->status !== 'disputed' || !$challenge->dispute) {
            return back()->with('error', 'Cannot upload evidence for this challenge.');
        }

        $request->validate([
            'evidence' => 'required|file|mimes:jpeg,png,jpg,gif,mp4,mov,avi|max:10240', // 10MB max
        ]);

        try {
            $dispute = $challenge->dispute;
            
            // Determine which evidence field to update
            $field = $user->id === $challenge->creator_id ? 'creator_evidence_path' : 'accepter_evidence_path';
            
            // Check if user has already uploaded evidence
            if (!empty($dispute->$field)) {
                return back()->with('error', 'You have already uploaded evidence for this dispute.');
            }
            
            // Store the evidence file
            $evidencePath = $request->file('evidence')->store('dispute-evidence', 'local');
            
            $dispute->update([
                $field => $evidencePath
            ]);

            // If both parties have uploaded evidence and admin hasn't been notified, notify admin
            if ($dispute->hasBothEvidence() && !$dispute->admin_notified) {
                // Here you would typically trigger a notification to admins
                // For now, we'll just mark as notified
                $dispute->markAdminNotified();
                
                return back()->with('success', 'Evidence uploaded successfully! Both parties have provided evidence. An admin will review and decide.');
            }

            return back()->with('success', 'Evidence uploaded successfully! Waiting for opponent to upload their evidence.');
        } catch (ValidationException $e) {
            return back()->withErrors($e->errors())->withInput();
        } catch (FileNotFoundException $e) {
            // Using Laravel's Log facade for proper logging
            Log::error('File not found during evidence upload: ' . $e->getMessage());
            return back()->with('error', 'The evidence file could not be found. Please try again.');
        } catch (\Exception $e) {
            // Using Laravel's Log facade for proper logging
            Log::error('Error uploading evidence: ' . $e->getMessage());
            $errorId = (string) Str::uuid();
            Log::error("Error ID {$errorId}: Error uploading evidence", [
                'exception' => $e,
            ]);
            return back()->with('error', "An error occurred while uploading evidence. Please try again. (Error ID: {$errorId})");
        }
    }

    /**
     * Open a dispute for a challenge.
     */
    public function dispute(Request $request, Challenge $challenge)
    {
        $user = Auth::user();

        if (!in_array($user->id, [$challenge->creator_id, $challenge->accepter_id])) {
            return back()->with('error', 'You are not part of this challenge.');
        }

        if ($challenge->status !== 'accepted') {
            return back()->with('error', 'Cannot dispute this challenge.');
        }

        $request->validate([
            'reason' => 'required|string|max:1000',
        ]);

        $challenge->dispute()->create([
            'opened_by' => $user->id,
            'reason' => $request->reason,
        ]);

        $challenge->update(['status' => 'disputed']);

        return redirect()->route('challenges.show', $challenge)
            ->with('success', 'Dispute opened. An admin will review your case.');
    }

    /**
     * Show challenge invite form
     */
    public function inviteForm(Challenge $challenge)
    {
        $user = Auth::user();

        // Only challenge creator can send invites
        if ($challenge->creator_id !== $user->id) {
            return back()->with('error', 'Only the challenge creator can send invites.');
        }

        if ($challenge->status !== 'open') {
            return back()->with('error', 'Cannot send invites for this challenge.');
        }

        // Get friends to invite
        $friends = $user->friends()->get();

        return view('challenges.invite', compact('challenge', 'friends'));
    }

    /**
     * Send challenge invites
     */
    public function sendInvites(Request $request, Challenge $challenge)
    {
        $user = Auth::user();

        // Only challenge creator can send invites
        if ($challenge->creator_id !== $user->id) {
            return back()->with('error', 'Only the challenge creator can send invites.');
        }

        if ($challenge->status !== 'open') {
            return back()->with('error', 'Cannot send invites for this challenge.');
        }

        $request->validate([
            'invitees' => 'required|array|min:1',
            'invitees.*' => 'exists:users,id',
            'message' => 'nullable|string|max:500',
        ]);

        $invitesSent = 0;
        $message = $request->message;

        foreach ($request->invitees as $inviteeId) {
            $invitee = \App\Models\User::find($inviteeId);
            
            if ($invitee && $challenge->inviteUser($invitee, $user, $message)) {
                $invitesSent++;
                // Send notification to invited user
                try {
                    $invitee->notify(new ChallengeInvitation($challenge, $user, $message));
                } catch (\Exception $e) {
                    \Log::error("Failed to send challenge invitation notification: " . $e->getMessage());
                }
            }
        }

        if ($invitesSent > 0) {
            return back()->with('success', "Sent {$invitesSent} challenge invite(s)!");
        }

        return back()->with('error', 'No invites were sent. Users may already be invited or ineligible.');
    }

    /**
     * Accept a challenge invite
     */
    public function acceptInvite(\App\Models\ChallengeInvite $invite)
    {
        $user = Auth::user();

        // Ensure this invite belongs to the current user
        if ($invite->invitee_id !== $user->id) {
            return back()->with('error', 'Unauthorized action.');
        }

        if (!$invite->canBeResponded()) {
            return back()->with('error', 'This invite is no longer valid.');
        }

        // Check if user can accept the challenge
        if (!$invite->challenge->canBeAcceptedBy($user)) {
            return back()->with('error', 'You cannot accept this challenge.');
        }

        if ($invite->accept()) {
            // Automatically accept the challenge
            return $this->accept(new Request(), $invite->challenge);
        }

        return back()->with('error', 'Unable to accept invite.');
    }

    /**
     * Decline a challenge invite
     */
    public function declineInvite(\App\Models\ChallengeInvite $invite)
    {
        $user = Auth::user();

        // Ensure this invite belongs to the current user
        if ($invite->invitee_id !== $user->id) {
            return back()->with('error', 'Unauthorized action.');
        }

        if ($invite->decline()) {
            return back()->with('success', "Challenge invite from {$invite->inviter->username} declined.");
        }

        return back()->with('error', 'Unable to decline invite.');
    }
}