<?php

namespace App\Http\Controllers\Creator;

use App\Http\Controllers\Controller;
use App\Models\Post;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class BlogController extends Controller
{
    /**
     * Ensure only creators can access these methods
     */
    public function __construct()
    {
        $this->middleware(['auth', function ($request, $next) {
            if (!Auth::user()->canPostBlogs()) {
                abort(403, 'Only creator-level users can submit blog posts.');
            }
            return $next($request);
        }]);
    }

    /**
     * Display creator's blog posts
     */
    public function index()
    {
        $posts = Auth::user()->posts()
            ->where('type', Post::TYPE_BLOG)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('creator.blogs.index', compact('posts'));
    }

    /**
     * Show the form for creating a new blog post
     */
    public function create()
    {
        return view('creator.blogs.create');
    }

    /**
     * Store a newly created blog post (pending approval)
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string|min:100',
            'excerpt' => 'nullable|string|max:500',
            'featured_image' => 'nullable|image|max:2048',
        ]);

        $slug = Post::generateSlug($request->title);

        $post = new Post([
            'title' => $request->title,
            'slug' => $slug,
            'content' => $request->content,
            'excerpt' => $request->excerpt,
            'type' => Post::TYPE_BLOG,
            'status' => Post::STATUS_DRAFT,
            'pending_approval' => true, // Mark as pending admin approval
            'author_id' => Auth::id(),
        ]);

        // Handle featured image upload
        if ($request->hasFile('featured_image')) {
            $path = $request->file('featured_image')->store('blog-images', 'public');
            $post->featured_image = $path;
        }

        $post->save();

        return redirect()->route('creator.blogs.index')
            ->with('success', 'Blog post submitted for admin approval!');
    }

    /**
     * Display the specified blog post
     */
    public function show(Post $post)
    {
        // Ensure the post belongs to the current user
        if ($post->author_id !== Auth::id()) {
            abort(403);
        }

        return view('creator.blogs.show', compact('post'));
    }

    /**
     * Show the form for editing a draft/pending blog post
     */
    public function edit(Post $post)
    {
        // Only allow editing of draft posts by the author (including pending approval)
        if ($post->author_id !== Auth::id() || $post->status !== Post::STATUS_DRAFT) {
            abort(403);
        }

        return view('creator.blogs.edit', compact('post'));
    }

    /**
     * Update the specified blog post
     */
    public function update(Request $request, Post $post)
    {
        // Only allow editing of draft posts by the author
        if ($post->author_id !== Auth::id() || $post->status !== Post::STATUS_DRAFT) {
            abort(403);
        }

        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string|min:100',
            'excerpt' => 'nullable|string|max:500',
            'featured_image' => 'nullable|image|max:2048',
        ]);

        $post->update([
            'title' => $request->title,
            'content' => $request->content,
            'excerpt' => $request->excerpt,
            'pending_approval' => true, // Mark as pending approval
        ]);

        // Handle featured image upload
        if ($request->hasFile('featured_image')) {
            $path = $request->file('featured_image')->store('blog-images', 'public');
            $post->featured_image = $path;
        }

        $post->save();

        return redirect()->route('creator.blogs.index')
            ->with('success', 'Blog post updated and submitted for approval!');
    }

    /**
     * Remove the specified blog post (only drafts)
     */
    public function destroy(Post $post)
    {
        if ($post->author_id !== Auth::id() || 
            ($post->status !== Post::STATUS_DRAFT || $post->pending_approval)) {
            abort(403, 'Can only delete drafts that are not pending approval.');
        }

        $post->delete();

        return redirect()->route('creator.blogs.index')
            ->with('success', 'Draft blog post deleted.');
    }
}
