<?php

namespace App\Http\Controllers;

use App\Models\Challenge;
use App\Models\Tournament;
use App\Models\User;
use App\Models\Dispute;
use App\Enums\UserRole;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    /**
     * Show the user dashboard for authenticated users.
     */
    public function index()
    {
        $user = auth()->user();
        
        // User performance statistics
        $userStats = [
            'total_challenges' => Challenge::where(function($q) use ($user) {
                $q->where('creator_id', $user->id)->orWhere('accepter_id', $user->id);
            })->where('status', 'completed')->count(),
            'wins' => Challenge::where(function($q) use ($user) {
                $q->where('creator_id', $user->id)->where('result', 'creator_win')
                 ->orWhere('accepter_id', $user->id)->where('result', 'accepter_win');
            })->count(),
            'total_earnings' => $user->walletTransactions()
                ->where('type', 'challenge_win')
                ->sum('amount'),
            'current_level' => $user->getCurrentLevel()?->name ?? 'Unranked',
            'role_display' => $user->role->getDisplayName(),
            'is_premium' => $user->role === UserRole::PREMIUM_USER,
            'is_moderator' => $user->role === UserRole::MODERATOR,
            'is_creator' => $user->role === UserRole::CREATOR,
        ];
        
        // Pending tournament invites
        $pendingTournamentInvites = $user->pendingTournamentInvites()
            ->with(['tournament.game', 'inviter'])
            ->limit(5)
            ->get();

        // Pending challenge invites
        $pendingChallengeInvites = $user->pendingChallengeInvites()
            ->with(['challenge.game', 'inviter'])
            ->limit(5)
            ->get();

        // Pending friend requests
        $pendingFriendRequests = $user->pendingFriendRequests()
            ->with('user')
            ->limit(5)
            ->get();

        // User's upcoming tournaments
        $userTournaments = $user->tournaments()
            ->where('status', 'upcoming')
            ->with(['game', 'host'])
            ->limit(3)
            ->get();

        // User's active challenges
        $userChallenges = Challenge::where(function($query) use ($user) {
            $query->where('creator_id', $user->id)
                  ->orWhere('accepter_id', $user->id);
        })
        ->whereIn('status', ['open', 'accepted'])
        ->with(['creator', 'accepter', 'game'])
        ->limit(3)
        ->get();

        // Recent activity for the user
        $recentActivity = Challenge::where(function($query) use ($user) {
            $query->where('creator_id', $user->id)
                  ->orWhere('accepter_id', $user->id);
        })
        ->where('status', 'completed')
        ->with(['creator', 'accepter', 'game'])
        ->latest()
        ->limit(5)
        ->get();

        // Role-specific data
        $moderatorStats = null;
        if ($user->canManageDisputes()) {
            $moderatorStats = [
                'pending_disputes' => $user->pendingModeratedDisputes()->count(),
                'total_disputes' => $user->moderatedDisputes()->count(),
                'unassigned_disputes' => Dispute::whereNull('moderator_id')->where('status', 'open')->count(),
            ];
        }

        return view('user-dashboard', compact(
            'userStats',
            'pendingTournamentInvites',
            'pendingChallengeInvites', 
            'pendingFriendRequests',
            'userTournaments',
            'userChallenges',
            'recentActivity',
            'moderatorStats'
        ));
    }
}