<?php

namespace App\Http\Controllers;

use App\Models\Friend;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

/**
 * FriendController
 * 
 * Handles all friend-related functionality including friend requests,
 * suggestions, and friend management. Implements robust error handling
 * and SQL optimization to prevent ambiguity issues.
 */
class FriendController extends Controller
{
    /**
     * Display the user's friends list
     * 
     * Shows accepted friends, pending friend requests received, and
     * friend requests sent by the current user.
     */
    public function index()
    {
        $user = Auth::user();
        $friends = $user->friends()->with('currentLevel')->paginate(20);
        $pendingRequests = $user->pendingFriendRequests()->with('user')->get();
        $sentRequests = $user->sentFriendRequests()->where('status', 'pending')->with('friend')->get();

        return view('friends.index', compact('friends', 'pendingRequests', 'sentRequests'));
    }

    /**
     * Send a friend request
     * 
     * Validates that users have played together before allowing friend requests.
     * Prevents spam and ensures meaningful connections.
     */
    public function sendRequest(Request $request, User $user)
    {
        $request->validate([
            'message' => 'nullable|string|max:255'
        ]);

        $currentUser = Auth::user();
        
        // Check if they've played together before - enforces platform rules
        if (!$currentUser->hasPlayedWith($user)) {
            return back()->with('error', 'You can only add players you\'ve played with as friends.');
        }

        $friendRequest = $currentUser->sendFriendRequest($user, $request->message);

        if ($friendRequest) {
            return back()->with('success', "Friend request sent to {$user->username}!");
        }

        return back()->with('error', 'Unable to send friend request. You may already be friends or have a pending request.');
    }

    /**
     * Accept a friend request
     */
    public function accept(Friend $friend)
    {
        $user = Auth::user();

        // Ensure this request belongs to the current user
        if ($friend->friend_id !== $user->id) {
            return back()->with('error', 'Unauthorized action.');
        }

        if ($friend->accept()) {
            return back()->with('success', "You are now friends with {$friend->user->username}!");
        }

        return back()->with('error', 'Unable to accept friend request.');
    }

    /**
     * Decline a friend request
     */
    public function decline(Friend $friend)
    {
        $user = Auth::user();

        // Ensure this request belongs to the current user
        if ($friend->friend_id !== $user->id) {
            return back()->with('error', 'Unauthorized action.');
        }

        if ($friend->decline()) {
            return back()->with('success', "Friend request from {$friend->user->username} declined.");
        }

        return back()->with('error', 'Unable to decline friend request.');
    }

    /**
     * Block a user
     */
    public function block(Friend $friend)
    {
        $user = Auth::user();

        // Ensure this involves the current user
        if ($friend->friend_id !== $user->id && $friend->user_id !== $user->id) {
            return back()->with('error', 'Unauthorized action.');
        }

        if ($friend->block()) {
            return back()->with('success', 'User blocked successfully.');
        }

        return back()->with('error', 'Unable to block user.');
    }

    /**
     * Remove a friend
     */
    public function remove(User $user)
    {
        $currentUser = Auth::user();

        // Find and remove the friendship from both sides
        Friend::where(function ($query) use ($currentUser, $user) {
            $query->where('user_id', $currentUser->id)->where('friend_id', $user->id);
        })->orWhere(function ($query) use ($currentUser, $user) {
            $query->where('user_id', $user->id)->where('friend_id', $currentUser->id);
        })->delete();

        return back()->with('success', "Removed {$user->username} from your friends list.");
    }

    /**
     * Show suggested friends (people user has played with)
     * 
     * Now accessible to all users, even those not logged in.
     * For logged-in users, suggests players they've competed against but aren't friends with.
     * Implements robust data type handling to prevent Collection diff errors.
     */
    public function suggestions()
    {
        $user = Auth::user();
        
        // If no user is logged in, show some public user suggestions
        if (!$user) {
            // For non-logged-in users, show active users with high levels as examples
            $suggestions = User::where('status', 'active')
                ->where('is_organizer', false)
                ->orderBy('xp', 'desc')
                ->with('currentLevel')
                ->paginate(20);
            
            return view('friends.suggestions', compact('suggestions'));
        }
        
        // Get users who have played with current user but aren't friends yet
        $playedWithUserIds = collect();
        
        // From challenges - find opponents from completed challenges
        $challengeUserIds = \App\Models\Challenge::where(function ($query) use ($user) {
            $query->where('creator_id', $user->id)->orWhere('accepter_id', $user->id);
        })->where('status', 'completed')
        ->get()
        ->map(function ($challenge) use ($user) {
            // Return the opponent's ID (not the current user's ID)
            return $challenge->creator_id === $user->id ? $challenge->accepter_id : $challenge->creator_id;
        })
        ->filter();

        // From tournaments - find other participants from completed tournaments
        $tournamentUserIds = \App\Models\Tournament::whereHas('participants', function ($query) use ($user) {
            $query->where('user_id', $user->id);
        })->where('status', 'completed')
        ->with('participants')
        ->get()
        ->flatMap(function ($tournament) use ($user) {
            return $tournament->participants->where('user_id', '!=', $user->id)->pluck('user_id');
        });

        $playedWithUserIds = $challengeUserIds->merge($tournamentUserIds)->unique();

        /**
         * CRITICAL FIX: Data type consistency for Collection diff operations
         * 
         * Problem: Laravel's Collection::diff() method expects all items to be objects 
         * with getKey() method, but we're working with integer IDs. This caused the
         * error "Call to a member function getKey() on int".
         * 
         * Solution: Convert all collections to integer arrays and use PHP's array_diff()
         * instead of Laravel's Collection diff() to ensure type consistency.
         */
        
        // Fix: Specify table name to resolve SQL ambiguity between users.id and friends.id
        // Since friends() returns User models, we want the users.id column
        // Convert all collections to simple arrays of integers to avoid getKey() errors
        $existingFriendIds = $user->friends()->pluck('users.id')->map(function($id) {
            return (int) $id;
        });
        $pendingRequestIds = $user->sentFriendRequests()->where('status', 'pending')->pluck('friend_id')->map(function($id) {
            return (int) $id;
        });
        $receivedRequestIds = $user->receivedFriendRequests()->where('status', 'pending')->pluck('user_id')->map(function($id) {
            return (int) $id;
        });

        // Fix: Ensure all collections contain integers for proper diff operation
        // Convert to array and use array_diff to avoid Collection diff issues with mixed data types
        $playedWithUserIdsArray = $playedWithUserIds->map(function($id) {
            return (int) $id;
        })->toArray();
        
        // Use PHP's array_diff instead of Collection diff to prevent getKey() errors
        $suggestedUserIdsArray = array_diff(
            $playedWithUserIdsArray,
            $existingFriendIds->toArray(),
            $pendingRequestIds->toArray(),
            $receivedRequestIds->toArray(),
            [$user->id] // Exclude self
        );
        $suggestedUserIds = collect($suggestedUserIdsArray);

        $suggestions = User::whereIn('id', $suggestedUserIds)
            ->with('currentLevel')
            ->paginate(20);

        return view('friends.suggestions', compact('suggestions'));
    }
}
