<?php

namespace App\Http\Controllers;

use App\Models\Challenge;
use App\Models\Tournament;
use App\Models\User;
use App\Models\Game;
use App\Models\Post;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    /**
     * Show the public home page.
     */
    public function index()
    {
        // Gaming platform statistics for impressive display
        $platformStats = [
            'total_players' => User::where('status', 'active')->count(),
            'total_challenges' => Challenge::count(),
            'total_tournaments' => Tournament::count(),
            'total_prize_pool' => Challenge::where('status', 'completed')->sum('wager_amount') * 2, // Total winnings distributed
            'active_competitions' => Challenge::whereIn('status', ['open', 'accepted'])->count() + 
                                   Tournament::whereIn('status', ['upcoming', 'in_progress'])->count(),
        ];

        // Featured tournaments with bigger prize pools
        $featuredTournaments = Tournament::with(['host', 'game', 'participants'])
            ->where('status', 'upcoming')
            ->orderBy('prize_pool', 'desc')
            ->limit(6)
            ->get();

        // High stakes challenges (show the most exciting ones)
        $highStakesChallenges = Challenge::with(['creator', 'game'])
            ->where('status', 'open')
            ->where('wager_amount', '>=', 1000) // Show challenges worth at least 1000
            ->orderBy('wager_amount', 'desc')
            ->limit(6)
            ->get();

        // Open challenges for general display
        $openChallenges = Challenge::with(['creator', 'game'])
            ->where('status', 'open')
            ->orderBy('created_at', 'desc')
            ->limit(12)
            ->get();

        // Top performing players this week
        $topPlayers = User::where('status', 'active')
            ->orderBy('xp', 'desc')
            ->limit(10)
            ->get();

        // Recent big wins for social proof
        $recentBigWins = Challenge::with(['creator', 'accepter', 'game'])
            ->where('status', 'completed')
            ->where('wager_amount', '>=', 500)
            ->latest()
            ->limit(5)
            ->get();

        // Recent activity (completed challenges for activity feed)
        $recentActivity = Challenge::with(['creator', 'accepter', 'game'])
            ->where('status', 'completed')
            ->latest()
            ->limit(10)
            ->get();

        // Popular games on the platform
        $popularGames = Game::withCount(['challenges', 'tournaments'])
            ->orderBy('challenges_count', 'desc')
            ->limit(8)
            ->get();

        // Latest gaming blog posts
        $latestBlogPosts = Post::published()
            ->where('type', Post::TYPE_BLOG)
            ->latest('published_at')
            ->limit(3)
            ->get();

        return view('dashboard', compact(
            'platformStats',
            'featuredTournaments',
            'highStakesChallenges',
            'openChallenges',
            'topPlayers',
            'recentBigWins',
            'recentActivity',
            'popularGames',
            'latestBlogPosts'
        ));
    }
}