<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Services\QoreIdService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class KycController extends Controller
{
    public function __construct(
        private QoreIdService $qoreIdService
    ) {}

    /**
     * Display KYC verification form
     */
    public function index()
    {
        $user = Auth::user();
        
        return view('kyc.index', compact('user'));
    }

    /**
     * Submit KYC verification request
     */
    public function submit(Request $request)
    {
        $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'date_of_birth' => 'required|date|before:today',
            'phone_number' => 'required|string|max:20',
            'address' => 'required|string|max:500',
            'id_type' => 'required|string|in:nin,bvn,drivers_license,international_passport',
            'id_number' => 'required|string|max:50',
            'id_document' => 'required|image|mimes:jpeg,png,jpg|max:2048',
            'selfie' => 'required|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $user = Auth::user();

        // Check if user already has pending or verified KYC
        if (in_array($user->kyc_status, ['pending', 'verified'])) {
            return back()->with('error', 'You already have a KYC verification request.');
        }

        try {
            // Upload documents
            $idDocumentPath = $request->file('id_document')->store('kyc/documents', 'public');
            $selfiePath = $request->file('selfie')->store('kyc/selfies', 'public');

            // Prepare data for QoreID API
            $kycData = [
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'date_of_birth' => $request->date_of_birth,
                'phone_number' => $request->phone_number,
                'address' => $request->address,
                'id_type' => $request->id_type,
                'id_number' => $request->id_number,
                'id_document_url' => asset('storage/' . $idDocumentPath),
                'selfie_url' => asset('storage/' . $selfiePath),
            ];

            // Submit to QoreID for verification
            $result = $this->qoreIdService->submitVerification($kycData);

            // Update user with KYC data and status
            $user->update([
                'kyc_status' => 'pending',
                'kyc_data' => json_encode($kycData),
                'qore_id_reference' => $result['reference'] ?? null,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'phone_number' => $request->phone_number,
                'date_of_birth' => $request->date_of_birth,
                'address' => $request->address,
            ]);

            return redirect()->route('kyc.status')
                ->with('success', 'KYC verification submitted successfully! We will review your documents within 24-48 hours.');

        } catch (\Exception $e) {
            Log::error('KYC submission failed: ' . $e->getMessage());
            
            return back()->with('error', 'Failed to submit KYC verification. Please try again later.')
                ->withInput();
        }
    }

    /**
     * Show KYC verification status
     */
    public function status()
    {
        $user = Auth::user();
        
        return view('kyc.status', compact('user'));
    }

    /**
     * Check verification status with QoreID
     */
    public function checkStatus()
    {
        $user = Auth::user();

        if (!$user->qore_id_reference) {
            return response()->json([
                'success' => false,
                'message' => 'No QoreID reference found'
            ]);
        }

        try {
            $result = $this->qoreIdService->checkStatus($user->qore_id_reference);
            
            // Update user status based on QoreID response
            if ($result['status'] === 'verified') {
                $user->update(['kyc_status' => 'verified']);
            } elseif ($result['status'] === 'rejected') {
                $user->update(['kyc_status' => 'rejected']);
            }

            return response()->json([
                'success' => true,
                'status' => $result['status'],
                'message' => $result['message'] ?? 'Status updated'
            ]);

        } catch (\Exception $e) {
            Log::error('KYC status check failed: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to check verification status'
            ]);
        }
    }

    /**
     * Webhook to receive QoreID status updates
     */
    public function webhook(Request $request)
    {
        try {
            // Verify webhook signature
            if (!$this->qoreIdService->verifyWebhook($request)) {
                return response()->json(['error' => 'Invalid signature'], 401);
            }

            $data = $request->all();
            $reference = $data['reference'] ?? null;
            $status = $data['status'] ?? null;

            if (!$reference || !$status) {
                return response()->json(['error' => 'Missing required fields'], 400);
            }

            // Find user by QoreID reference
            $user = User::where('qore_id_reference', $reference)->first();
            
            if (!$user) {
                return response()->json(['error' => 'User not found'], 404);
            }

            // Update user KYC status
            $kycStatus = match($status) {
                'verified', 'approved' => 'verified',
                'rejected', 'failed' => 'rejected',
                default => 'pending'
            };

            $user->update(['kyc_status' => $kycStatus]);

            Log::info("KYC status updated via webhook: User {$user->id}, Status: {$kycStatus}");

            return response()->json(['success' => true]);

        } catch (\Exception $e) {
            Log::error('KYC webhook failed: ' . $e->getMessage());
            
            return response()->json(['error' => 'Webhook processing failed'], 500);
        }
    }
}