<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\WalletTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class LeaderboardController extends Controller
{
    /**
     * Display the leaderboards.
     */
    public function index(Request $request)
    {
        $period = $request->get('period', 'all_time');
        $type = $request->get('type', 'wins');

        // Define date ranges for different periods
        $dateRange = $this->getDateRangeForPeriod($period);

        switch ($type) {
            case 'winnings':
                $users = $this->getWinningsLeaderboard($period, $dateRange);
                break;
            case 'wins':
                $users = $this->getWinsLeaderboard($period, $dateRange);
                break;
            case 'xp':
                // XP leaderboard doesn't use time periods as XP is cumulative
                $users = User::where('status', 'active')
                    ->orderBy('xp', 'desc')
                    ->limit(50)
                    ->get();
                break;
            default:
                $users = $this->getWinsLeaderboard($period, $dateRange);
        }

        // Return JSON for AJAX requests
        if ($request->ajax()) {
            return response()->json([
                'users' => $users,
                'period' => $period,
                'type' => $type,
                'timestamp' => now()->format('H:i:s')
            ]);
        }

        return view('leaderboards.index', compact('users', 'period', 'type'));
    }

    /**
     * Get date range for a given period
     */
    private function getDateRangeForPeriod(string $period): array
    {
        switch ($period) {
            case 'daily':
                return [
                    'start' => today()->startOfDay(),
                    'end' => today()->endOfDay(),
                ];
            case 'weekly':
                return [
                    'start' => now()->startOfWeek(),
                    'end' => now()->endOfWeek(),
                ];
            case 'monthly':
                return [
                    'start' => now()->startOfMonth(),
                    'end' => now()->endOfMonth(),
                ];
            default: // all_time
                return [
                    'start' => null,
                    'end' => null,
                ];
        }
    }

    /**
     * Get wins leaderboard for a specific period
     */
    private function getWinsLeaderboard(string $period, array $dateRange)
    {
        if ($period === 'all_time') {
            return $this->getAllTimeWinsLeaderboard();
        } else {
            return $this->getTimePeriodWinsLeaderboard($dateRange);
        }
    }

    /**
     * Get all-time wins leaderboard optimized for MySQL
     */
    private function getAllTimeWinsLeaderboard()
    {
        // Use MySQL-optimized subquery with database-level filtering for better performance
        return User::select([
            'users.id', 'users.username', 'users.level', 'users.xp', 'users.avatar_url',
            DB::raw('(
                COALESCE((SELECT COUNT(*) FROM challenges c1 WHERE c1.creator_id = users.id AND c1.result = "creator_win" AND c1.status = "completed"), 0) +
                COALESCE((SELECT COUNT(*) FROM challenges c2 WHERE c2.accepter_id = users.id AND c2.result = "accepter_win" AND c2.status = "completed"), 0)
            ) as total_wins')
        ])
        ->from(DB::raw('(
            SELECT users.* FROM users 
            WHERE users.status = "active" 
            AND (
                (SELECT COUNT(*) FROM challenges c1 WHERE c1.creator_id = users.id AND c1.result = "creator_win" AND c1.status = "completed") +
                (SELECT COUNT(*) FROM challenges c2 WHERE c2.accepter_id = users.id AND c2.result = "accepter_win" AND c2.status = "completed")
            ) > 0
        ) as users'))
        ->orderByDesc('total_wins')
        ->limit(50)
        ->get();
    }

    /**
     * Build MySQL-optimized SQL query for time-period wins leaderboard
     */
    private function buildTimePeriodWinsQuery(): string
    {
        return "
            SELECT users.id, users.username, users.level, users.xp, users.avatar_url,
                (
                    COALESCE((SELECT COUNT(*) FROM challenges c1 
                             WHERE c1.creator_id = users.id 
                             AND c1.result = 'creator_win' 
                             AND c1.status = 'completed'
                             AND c1.updated_at BETWEEN ? AND ?), 0) +
                    COALESCE((SELECT COUNT(*) FROM challenges c2 
                             WHERE c2.accepter_id = users.id 
                             AND c2.result = 'accepter_win' 
                             AND c2.status = 'completed'
                             AND c2.updated_at BETWEEN ? AND ?), 0)
                ) as total_wins
            FROM (
                SELECT users.* FROM users 
                WHERE users.status = 'active' 
                AND (
                    (SELECT COUNT(*) FROM challenges c1 
                     WHERE c1.creator_id = users.id 
                     AND c1.result = 'creator_win' 
                     AND c1.status = 'completed'
                     AND c1.updated_at BETWEEN ? AND ?) +
                    (SELECT COUNT(*) FROM challenges c2 
                     WHERE c2.accepter_id = users.id 
                     AND c2.result = 'accepter_win' 
                     AND c2.status = 'completed'
                     AND c2.updated_at BETWEEN ? AND ?)
                ) > 0
            ) as users
            ORDER BY total_wins DESC
            LIMIT 50
        ";
    }

    /**
     * Get time-period wins leaderboard using raw SQL for date filtering
     */
    private function getTimePeriodWinsLeaderboard(array $dateRange)
    {
        $sql = $this->buildTimePeriodWinsQuery();
        
        $users = DB::select($sql, [
            $dateRange['start'], 
            $dateRange['end'], 
            $dateRange['start'], 
            $dateRange['end'],
            $dateRange['start'], 
            $dateRange['end'], 
            $dateRange['start'], 
            $dateRange['end']
        ]);
        
        return collect($users);
    }

    /**
     * Get winnings leaderboard for a specific period
     */
    private function getWinningsLeaderboard(string $period, array $dateRange)
    {
        if ($period === 'all_time') {
            // Use database-level aggregation for better performance
            $users = User::select([
                'users.id', 'users.username', 'users.level', 'users.xp', 'users.avatar_url',
                DB::raw('COALESCE(SUM(wt.amount), 0) as total_winnings')
            ])
            ->leftJoin('wallet_transactions as wt', function($join) {
                $join->on('users.id', '=', 'wt.user_id')
                     ->where('wt.type', '=', 'prize')
                     ->where('wt.status', '=', 'completed');
            })
            ->where('users.status', 'active')
            ->groupBy('users.id', 'users.username', 'users.level', 'users.xp', 'users.avatar_url')
            ->havingRaw('total_winnings > 0')
            ->orderByDesc('total_winnings')
            ->limit(50)
            ->get();
        } else {
            // For time-based periods, filter by transaction date at database level
            $users = User::select([
                'users.id', 'users.username', 'users.level', 'users.xp', 'users.avatar_url',
                DB::raw('COALESCE(SUM(wt.amount), 0) as total_winnings')
            ])
            ->leftJoin('wallet_transactions as wt', function($join) use ($dateRange) {
                $join->on('users.id', '=', 'wt.user_id')
                     ->where('wt.type', '=', 'prize')
                     ->where('wt.status', '=', 'completed')
                     ->whereBetween('wt.created_at', [$dateRange['start'], $dateRange['end']]);
            })
            ->where('users.status', 'active')
            ->groupBy('users.id', 'users.username', 'users.level', 'users.xp', 'users.avatar_url')
            ->havingRaw('total_winnings > 0')
            ->orderByDesc('total_winnings')
            ->limit(50)
            ->get();
        }

        return $users;
    }
}