<?php

namespace App\Http\Controllers\Moderator;

use App\Http\Controllers\Controller;
use App\Models\Dispute;
use App\Services\WalletService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class DisputeController extends Controller
{
    public function __construct(
        private WalletService $walletService
    ) {
        // Ensure only moderators can access these routes
        $this->middleware(function ($request, $next) {
            if (!Auth::user()->canManageDisputes()) {
                abort(403, 'Unauthorized. Only moderators can access this area.');
            }
            return $next($request);
        });
    }

    /**
     * Display disputes assigned to this moderator
     */
    public function index()
    {
        $user = Auth::user();
        
        // Get disputes assigned to this moderator
        $disputes = Dispute::where('moderator_id', $user->id)
            ->with(['challenge.creator', 'challenge.accepter', 'challenge.game', 'openedBy'])
            ->latest()
            ->paginate(20);

        return view('moderator.disputes.index', compact('disputes'));
    }

    /**
     * Display all disputes for moderator overview
     */
    public function all()
    {
        // Only show unassigned disputes and disputes assigned to this moderator
        $user = Auth::user();
        
        $disputes = Dispute::where(function($query) use ($user) {
                $query->where('moderator_id', $user->id)
                      ->orWhereNull('moderator_id');
            })
            ->with(['challenge.creator', 'challenge.accepter', 'challenge.game', 'openedBy', 'moderator'])
            ->latest()
            ->paginate(20);

        return view('moderator.disputes.all', compact('disputes'));
    }

    /**
     * Claim an unassigned dispute
     */
    public function claim(Dispute $dispute)
    {
        // Check if dispute is unassigned
        if ($dispute->hasAssignedModerator()) {
            return redirect()->back()->with('error', 'This dispute is already assigned to another moderator.');
        }

        // Assign to current moderator
        $dispute->assignToModerator(Auth::user());

        return redirect()->back()->with('success', 'Dispute claimed successfully!');
    }

    /**
     * Display the specified dispute
     */
    public function show(Dispute $dispute)
    {
        // Check if moderator can view this dispute
        if ($dispute->moderator_id !== Auth::id() && !$dispute->moderator_id) {
            abort(403, 'You can only view disputes assigned to you.');
        }

        $dispute->load([
            'challenge.creator',
            'challenge.accepter',
            'challenge.game',
            'challenge.matchProofs.user',
            'openedBy',
            'moderator'
        ]);

        return view('moderator.disputes.show', compact('dispute'));
    }

    /**
     * Resolve a dispute (moderator decision)
     */
    public function decide(Request $request, Dispute $dispute)
    {
        // Check if moderator can resolve this dispute
        if ($dispute->moderator_id !== Auth::id()) {
            abort(403, 'You can only resolve disputes assigned to you.');
        }

        $request->validate([
            'decision' => 'required|in:creator_win,accepter_win,refund',
            'resolution' => 'required|string|max:1000',
        ]);

        $challenge = $dispute->challenge;

        // Update dispute with moderator resolution
        $dispute->update([
            'status' => 'resolved',
            'resolution' => $request->resolution,
            'admin_id' => Auth::id(), // Track who resolved it (moderator acts as admin for this)
        ]);

        // Update challenge based on decision
        if ($request->decision === 'refund') {
            $challenge->update([
                'status' => 'completed',
                'result' => 'none',
            ]);
            
            // Refund both players
            $this->walletService->refundChallenge($challenge);
        } else {
            $challenge->update([
                'status' => 'completed',
                'result' => $request->decision,
            ]);
            
            // Process payout to winner
            $this->walletService->processChallengeWin($challenge);
        }

        return redirect()->route('moderator.disputes.index')
            ->with('success', 'Dispute resolved successfully!');
    }

    /**
     * Dashboard for moderators
     */
    public function dashboard()
    {
        $user = Auth::user();
        
        $stats = [
            'assigned_disputes' => $user->moderatedDisputes()->count(),
            'pending_disputes' => $user->pendingModeratedDisputes()->count(),
            'resolved_disputes' => $user->moderatedDisputes()->where('status', 'resolved')->count(),
            'unassigned_disputes' => Dispute::whereNull('moderator_id')->where('status', 'open')->count(),
        ];

        // Recent disputes
        $recentDisputes = $user->moderatedDisputes()
            ->with(['challenge.creator', 'challenge.accepter', 'challenge.game'])
            ->latest()
            ->limit(5)
            ->get();

        return view('moderator.dashboard', compact('stats', 'recentDisputes'));
    }
}