<?php

namespace App\Http\Controllers;

use App\Models\Notification;
use App\Models\PopupNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class NotificationController extends Controller
{
    /**
     * Display user's notification center
     */
    public function index()
    {
        $notifications = Auth::user()
                            ->notifications()
                            ->latest('created_at')
                            ->paginate(20);

        // Mark all as read when viewing
        Auth::user()->notifications()
             ->whereNull('read_at')
             ->update(['read_at' => now()]);

        return view('notifications.index', compact('notifications'));
    }

    /**
     * Get pending popup notifications for user
     */
    public function getPopups()
    {
        $user = Auth::user();
        
        $popupNotifications = PopupNotification::active()
                                             ->get()
                                             ->filter(function ($notification) use ($user) {
                                                 return $notification->shouldShowToUser($user);
                                             });

        return response()->json($popupNotifications->values());
    }

    /**
     * Mark popup notification as seen
     */
    public function markPopupSeen(Request $request, PopupNotification $popupNotification)
    {
        $popupNotification->markAsSeenBy(Auth::user());

        return response()->json(['success' => true]);
    }

    /**
     * Mark notification as read
     */
    public function markAsRead(Notification $notification)
    {
        // Ensure user owns this notification
        if ($notification->user_id !== Auth::id()) {
            abort(403);
        }

        $notification->markAsRead();

        return response()->json(['success' => true]);
    }

    /**
     * Get unread notification count
     */
    public function getUnreadCount()
    {
        $count = Auth::user()
                    ->notifications()
                    ->whereNull('read_at')
                    ->count();

        return response()->json(['count' => $count]);
    }

    /**
     * Mark all notifications as read
     */
    public function markAllAsRead()
    {
        Auth::user()
            ->notifications()
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        return redirect()->back()->with('success', 'All notifications marked as read!');
    }
}
