<?php

namespace App\Http\Controllers;

use App\Models\Post;
use Illuminate\Http\Request;

class PostController extends Controller
{
    /**
     * Display a listing of posts
     */
    public function index(Request $request)
    {
        $query = Post::with('author')
                    ->published()
                    ->latest('published_at');

        // Filter by type
        if ($request->filled('type')) {
            $query->ofType($request->type);
        }

        // Search by title and content
        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('title', 'like', '%' . $request->search . '%')
                  ->orWhere('content', 'like', '%' . $request->search . '%');
            });
        }

        $posts = $query->paginate(12);

        // Get post types for filtering
        $postTypes = [
            Post::TYPE_BLOG => 'Blog',
            Post::TYPE_NEWS => 'News',
            Post::TYPE_HIGHLIGHT => 'Highlights',
            Post::TYPE_TOURNAMENT => 'Tournaments',
            Post::TYPE_UPDATE => 'Updates',
        ];

        return view('posts.index', compact('posts', 'postTypes'));
    }

    /**
     * Display the specified post
     */
    public function show($slug)
    {
        $post = Post::with('author')
                   ->where('slug', $slug)
                   ->published()
                   ->firstOrFail();

        // Increment view count
        $post->incrementViews();

        // Get related posts
        $relatedPosts = Post::with('author')
                           ->published()
                           ->where('type', $post->type)
                           ->where('id', '!=', $post->id)
                           ->latest('published_at')
                           ->limit(3)
                           ->get();

        return view('posts.show', compact('post', 'relatedPosts'));
    }

    /**
     * Display posts by type
     */
    public function byType($type)
    {
        if (!in_array($type, [Post::TYPE_BLOG, Post::TYPE_NEWS, Post::TYPE_HIGHLIGHT, Post::TYPE_TOURNAMENT, Post::TYPE_UPDATE])) {
            abort(404);
        }

        $posts = Post::with('author')
                    ->published()
                    ->ofType($type)
                    ->latest('published_at')
                    ->paginate(12);

        $typeLabels = [
            Post::TYPE_BLOG => 'Blog Posts',
            Post::TYPE_NEWS => 'News Updates',
            Post::TYPE_HIGHLIGHT => 'Tournament Highlights',
            Post::TYPE_TOURNAMENT => 'Tournament Announcements',
            Post::TYPE_UPDATE => 'Platform Updates',
        ];

        $title = $typeLabels[$type];

        return view('posts.type', compact('posts', 'type', 'title'));
    }
}
