<?php

namespace App\Http\Controllers;

use App\Models\Tournament;
use App\Models\TournamentInvite;
use App\Models\Game;
use App\Models\User;
use App\Notifications\TournamentInvitation;
use App\Services\WalletService;
use App\Services\XPService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class TournamentController extends Controller
{
    public function __construct(
        private WalletService $walletService,
        private XPService $xpService
    ) {}

    /**
     * Display a listing of tournaments.
     */
    public function index(Request $request)
    {
        $filter = $request->get('filter', 'global'); // 'global' or 'friends'
        $gameId = $request->get('game_id');
        $status = $request->get('status', 'upcoming,ongoing');
        $type = $request->get('type');

        $query = Tournament::with(['host', 'game', 'participants']);

        // Apply status filter
        $statuses = explode(',', $status);
        $query->whereIn('status', $statuses);

        // Apply friend filter if authenticated and requested
        if (auth()->check() && $filter === 'friends') {
            $friendIds = auth()->user()->friends()->pluck('users.id');
            $query->whereIn('host_id', $friendIds);
        }

        // Apply game filter
        if ($gameId) {
            $query->where('game_id', $gameId);
        }

        // Apply type filter
        if ($type) {
            $query->where('type', $type);
        }

        $tournaments = $query->latest()->paginate(15);
        $games = Game::all();

        return view('tournaments.index', compact('tournaments', 'games', 'filter'));
    }

    /**
     * Display the user's tournaments (participant or host).
     */
    public function myTournaments(Request $request)
    {
        $user = auth()->user();
        $filter = $request->get('filter', 'all'); // 'all', 'ongoing', 'past'
        $gameId = $request->get('game_id');

        // Get tournaments where user is host or participant
        $query = Tournament::with(['host', 'game', 'participants'])
            ->where(function($q) use ($user) {
                $q->where('host_id', $user->id)
                  ->orWhereHas('participants', function($subQuery) use ($user) {
                      $subQuery->where('user_id', $user->id);
                  });
            });

        // Apply status filter
        switch ($filter) {
            case 'ongoing':
                $query->whereIn('status', ['upcoming', 'ongoing']);
                break;
            case 'past':
                $query->whereIn('status', ['completed', 'cancelled']);
                break;
            default:
                // Show all (ongoing and past)
                $query->whereIn('status', ['upcoming', 'ongoing', 'completed', 'cancelled']);
                break;
        }

        // Apply game filter
        if ($gameId) {
            $query->where('game_id', $gameId);
        }

        $tournaments = $query->latest()->paginate(15);
        $games = Game::all();

        return view('tournaments.my', compact('tournaments', 'games', 'filter'));
    }

    /**
     * Show the form for creating a new tournament.
     */
    public function create()
    {
        $this->authorize('create', Tournament::class);
        
        $games = Game::all();
        return view('tournaments.create', compact('games'));
    }

    /**
     * Store a newly created tournament in storage.
     */
    public function store(Request $request)
    {
        $this->authorize('create', Tournament::class);

        $user = Auth::user();
        
        // Check if user's role allows tournament creation
        if (!$user->canCreateTournaments()) {
            return back()->withErrors([
                'general' => "Your role ({$user->role}) does not allow tournament creation. Only highest and creator level users can create tournaments."
            ])->withInput();
        }

        $request->validate([
            'game_id' => 'required|exists:games,id',
            'type' => 'required|in:knockout,round_robin',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'entry_fee' => 'required|numeric|min:0',
            'livestream_link' => 'nullable|url',
            'starts_at' => 'required|date|after:now',
        ]);

        $tournament = Tournament::create([
            'host_id' => Auth::id(),
            'game_id' => $request->game_id,
            'type' => $request->type,
            'title' => $request->title,
            'description' => $request->description,
            'entry_fee' => $request->entry_fee,
            'livestream_link' => $request->livestream_link,
            'starts_at' => $request->starts_at,
        ]);

        // Award XP for creating tournament
        $this->xpService->awardXP($user, 15, 'tournament_created');

        return redirect()->route('tournaments.show', $tournament)
            ->with('success', 'Tournament created successfully!');
    }

    /**
     * Display the specified tournament.
     */
    public function show(Tournament $tournament)
    {
        $tournament->load(['host', 'game', 'participants.user', 'matches.playerA', 'matches.playerB']);
        
        return view('tournaments.show', compact('tournament'));
    }

    /**
     * Join a tournament.
     */
    public function join(Tournament $tournament)
    {
        $user = Auth::user();

        if (!$tournament->canBeJoinedBy($user)) {
            return back()->with('error', 'You cannot join this tournament.');
        }

        // Charge entry fee
        if ($tournament->entry_fee > 0) {
            $this->walletService->lockFunds($user, $tournament->entry_fee, 'tournament', $tournament->id);
        }

        $tournament->participants()->create([
            'user_id' => $user->id,
        ]);

        // Award XP for joining tournament
        $this->xpService->awardXP($user, 15, 'tournament_joined');

        return redirect()->route('tournaments.show', $tournament)
            ->with('success', 'Successfully joined the tournament!');
    }

    /**
     * Show invitation form for a tournament.
     */
    public function inviteForm(Tournament $tournament)
    {
        $this->authorize('update', $tournament);

        // Get users that can be invited (not already participating or invited)
        $excludedUserIds = $tournament->users()->pluck('user_id')
            ->merge($tournament->invites()->pluck('invitee_id'))
            ->push(Auth::id()); // Exclude current user

        $availableUsers = User::whereNotIn('id', $excludedUserIds)
            ->where('status', 'active')
            ->orderBy('username')
            ->get();

        return view('tournaments.invite', compact('tournament', 'availableUsers'));
    }

    /**
     * Send tournament invitations.
     */
    public function sendInvites(Request $request, Tournament $tournament)
    {
        $this->authorize('update', $tournament);

        $request->validate([
            'user_ids' => 'required|array|min:1',
            'user_ids.*' => 'exists:users,id',
            'message' => 'nullable|string|max:500'
        ]);

        $invitedCount = 0;
        $errors = [];

        foreach ($request->user_ids as $userId) {
            $user = User::find($userId);
            
            if ($tournament->hasUserInviteOrParticipation($user)) {
                $errors[] = "{$user->username} is already invited or participating";
                continue;
            }

            if ($tournament->hasParticipantLimit()) {
                $errors[] = "Tournament has reached participant limit";
                break;
            }

            $invite = $tournament->inviteUser($user, Auth::user(), $request->message);
            if ($invite) {
                $invitedCount++;
                // Send notification to invited user
                try {
                    $user->notify(new TournamentInvitation($tournament, Auth::user(), $request->message));
                } catch (\Exception $e) {
                    \Log::error("Failed to send tournament invitation notification: " . $e->getMessage());
                }
            } else {
                $errors[] = "Failed to invite {$user->username}";
            }
        }

        $message = "Successfully sent {$invitedCount} invitations.";
        if (!empty($errors)) {
            $message .= " Errors: " . implode(', ', $errors);
        }

        return redirect()->route('tournaments.show', $tournament)
            ->with($invitedCount > 0 ? 'success' : 'error', $message);
    }

    /**
     * Accept a tournament invitation.
     */
    public function acceptInvite(TournamentInvite $invite)
    {
        $user = Auth::user();

        if ($invite->invitee_id !== $user->id) {
            abort(403, 'Unauthorized');
        }

        if (!$invite->canBeResponded()) {
            return back()->with('error', 'This invitation has expired or already been responded to.');
        }

        $tournament = $invite->tournament;

        if (!$tournament->canBeJoinedBy($user)) {
            return back()->with('error', 'You cannot join this tournament anymore.');
        }

        // Accept the invite
        $invite->accept();

        // Join the tournament
        if ($tournament->entry_fee > 0) {
            $this->walletService->lockFunds($user, $tournament->entry_fee, 'tournament', $tournament->id);
        }

        $tournament->participants()->create([
            'user_id' => $user->id,
        ]);

        // Award XP for joining tournament
        $this->xpService->awardXP($user, 15, 'tournament_joined');

        return redirect()->route('tournaments.show', $tournament)
            ->with('success', 'Successfully joined the tournament!');
    }

    /**
     * Decline a tournament invitation.
     */
    public function declineInvite(TournamentInvite $invite)
    {
        $user = Auth::user();

        if ($invite->invitee_id !== $user->id) {
            abort(403, 'Unauthorized');
        }

        if (!$invite->canBeResponded()) {
            return back()->with('error', 'This invitation has expired or already been responded to.');
        }

        $invite->decline();

        return redirect()->route('dashboard')
            ->with('success', 'Tournament invitation declined.');
    }

    /**
     * Start a tournament.
     */
    public function start(Tournament $tournament)
    {
        $this->authorize('update', $tournament);

        if (!$tournament->canStart()) {
            return back()->with('error', 'Tournament cannot be started. Check participant requirements.');
        }

        if ($tournament->start()) {
            return redirect()->route('tournaments.show', $tournament)
                ->with('success', 'Tournament started successfully!');
        }

        return back()->with('error', 'Failed to start tournament.');
    }

    /**
     * Report a tournament match result.
     */
    public function reportMatch(Request $request, Tournament $tournament)
    {
        $user = Auth::user();

        $request->validate([
            'match_id' => 'required|exists:tournament_matches,id',
            'result' => 'required|in:player_a_win,player_b_win',
        ]);

        $match = $tournament->matches()->findOrFail($request->match_id);

        // Verify user is part of the match
        if (!in_array($user->id, [$match->player_a_id, $match->player_b_id])) {
            return back()->with('error', 'You are not part of this match.');
        }

        $match->update(['result' => $request->result]);

        // Award XP to winner
        $winner = $match->getWinner();
        if ($winner) {
            $this->xpService->awardXP($winner, 20, 'tournament_match_won');
        }

        return redirect()->route('tournaments.show', $tournament)
            ->with('success', 'Match result reported successfully!');
    }
}