<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\WalletTransaction;
use App\Services\WalletService;
use App\Services\CreditTransferService;
use Illuminate\Http\Request;
use App\Services\PaymentService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class WalletController extends Controller
{
    public function __construct(
        private WalletService $walletService,
        private PaymentService $paymentService,
        private CreditTransferService $creditTransferService
    ) {}

    /**
     * Display the user's wallet.
     */
    public function index()
    {
        $user = Auth::user();
        $transactions = $user->walletTransactions()
            ->latest()
            ->paginate(20);

        return view('wallet.index', compact('user', 'transactions'));
    }

    /**
     * Show the deposit form.
     */
    public function showDeposit()
    {
        return view('wallet.deposit');
    }

    /**
     * Process a deposit request.
     */
    public function deposit(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:10|max:10000',
            'provider' => 'required|in:paystack,flutterwave,paypal,stripe',
        ]);

        try {
            $transaction = $this->walletService->initiateDeposit(
                Auth::user(),
                $request->amount,
                $request->provider
            );

            // Use real payment service
            $paymentResult = $this->paymentService->initiateDeposit($transaction);

            if ($paymentResult['success']) {
                return redirect($paymentResult['payment_url']);
            }

            return redirect()->route('wallet.index')
                ->with('error', 'Failed to initiate payment. Please try again.');

        } catch (\Exception $e) {
            return redirect()->route('wallet.index')
                ->with('error', 'Payment processing error: ' . $e->getMessage());
        }
    }

    /**
     * Handle payment callback from providers.
     */
    public function handleCallback(Request $request, string $provider)
    {
        try {
            $reference = $request->input('reference') ?? $request->input('tx_ref') ?? $request->input('trxref');
            
            if (!$reference) {
                return redirect()->route('wallet.index')
                    ->with('error', 'Invalid payment reference.');
            }

            $verification = $this->paymentService->verifyPayment($provider, $reference);

            if ($verification['success']) {
                $transaction = WalletTransaction::where('reference', $reference)->first();
                
                if ($transaction && $transaction->status === 'pending') {
                    $this->walletService->completeDeposit($transaction);
                    
                    return redirect()->route('wallet.index')
                        ->with('success', 'Deposit completed successfully!');
                }
            }

            return redirect()->route('wallet.index')
                ->with('error', 'Payment verification failed.');

        } catch (\Exception $e) {
            return redirect()->route('wallet.index')
                ->with('error', 'Payment processing error: ' . $e->getMessage());
        }
    }

    /**
     * Show the withdrawal form.
     */
    public function showWithdraw()
    {
        return view('wallet.withdraw');
    }

    /**
     * Process a withdrawal request.
     */
    public function withdraw(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'amount' => 'required|numeric|min:10|max:' . $user->wallet_balance,
            'provider' => 'required|in:paystack,flutterwave,paypal,stripe',
            'account_details' => 'required|string',
        ]);

        try {
            $transaction = $this->walletService->initiateWithdrawal(
                $user,
                $request->amount,
                $request->provider,
                ['account_details' => $request->account_details]
            );

            // In a real implementation, process the withdrawal
            // For now, mark as pending for admin approval
            return redirect()->route('wallet.index')
                ->with('success', 'Withdrawal request submitted successfully! It will be processed within 24 hours.');

        } catch (\Exception $e) {
            return redirect()->route('wallet.index')
                ->with('error', 'Withdrawal processing error: ' . $e->getMessage());
        }
    }

    /**
     * Show transaction details.
     */
    public function showTransaction(WalletTransaction $transaction)
    {
        if ($transaction->user_id !== Auth::id()) {
            abort(403);
        }

        return view('wallet.transaction', compact('transaction'));
    }

    /**
     * Show PIN setup form.
     */
    public function showPinSetup()
    {
        $user = Auth::user();
        
        if ($user->hasTransactionPin()) {
            return redirect()->route('wallet.index')
                ->with('info', 'You already have a transaction PIN set. Use the change PIN option to update it.');
        }

        return view('wallet.pin-setup');
    }

    /**
     * Set transaction PIN.
     */
    public function setupPin(Request $request)
    {
        $user = Auth::user();

        if ($user->hasTransactionPin()) {
            return redirect()->route('wallet.index')
                ->with('error', 'Transaction PIN already set.');
        }

        $request->validate([
            'pin' => 'required|digits:4',
            'pin_confirmation' => 'required|same:pin',
            'password' => 'required|string',
        ]);

        // Verify current password
        if (!Hash::check($request->password, $user->password)) {
            return back()->withErrors(['password' => 'Current password is incorrect.']);
        }

        $user->setTransactionPin($request->pin);

        return redirect()->route('wallet.index')
            ->with('success', 'Transaction PIN set successfully!');
    }

    /**
     * Show PIN change form.
     */
    public function showPinChange()
    {
        $user = Auth::user();
        
        if (!$user->hasTransactionPin()) {
            return redirect()->route('wallet.pin-setup')
                ->with('info', 'Please set up your transaction PIN first.');
        }

        return view('wallet.pin-change');
    }

    /**
     * Change transaction PIN.
     */
    public function changePin(Request $request)
    {
        $user = Auth::user();

        if (!$user->hasTransactionPin()) {
            return redirect()->route('wallet.pin-setup');
        }

        $request->validate([
            'current_pin' => 'required|digits:4',
            'new_pin' => 'required|digits:4',
            'new_pin_confirmation' => 'required|same:new_pin',
            'password' => 'required|string',
        ]);

        // Verify current password
        if (!Hash::check($request->password, $user->password)) {
            return back()->withErrors(['password' => 'Current password is incorrect.']);
        }

        // Verify current PIN
        if (!$user->verifyTransactionPin($request->current_pin)) {
            return back()->withErrors(['current_pin' => 'Current PIN is incorrect.']);
        }

        $user->setTransactionPin($request->new_pin);

        return redirect()->route('wallet.index')
            ->with('success', 'Transaction PIN changed successfully!');
    }

    /**
     * Show credit transfer form.
     */
    public function showTransfer()
    {
        $user = Auth::user();
        
        if (!$user->hasTransactionPin()) {
            return redirect()->route('wallet.pin-setup')
                ->with('info', 'Please set up your transaction PIN first before transferring credits.');
        }

        return view('wallet.transfer');
    }

    /**
     * Process credit transfer.
     */
    public function transfer(Request $request)
    {
        $user = Auth::user();

        if (!$user->hasTransactionPin()) {
            return redirect()->route('wallet.pin-setup');
        }

        $request->validate([
            'recipient_username' => 'required|string|exists:users,username',
            'amount' => 'required|numeric|min:1|max:100000',
            'note' => 'nullable|string|max:255',
            'pin' => 'required|digits:4',
        ]);

        // Find recipient
        $recipient = User::where('username', $request->recipient_username)->first();

        if (!$recipient) {
            return back()->withErrors(['recipient_username' => 'User not found.']);
        }

        if ($recipient->id === $user->id) {
            return back()->withErrors(['recipient_username' => 'Cannot transfer to yourself.']);
        }

        if ($recipient->status === 'banned') {
            return back()->withErrors(['recipient_username' => 'Cannot transfer to banned user.']);
        }

        try {
            $transfer = $this->creditTransferService->initiateTransfer(
                $user,
                $recipient,
                $request->amount,
                $request->pin,
                $request->note
            );

            return redirect()->route('wallet.transfer.show', $transfer)
                ->with('success', 'Credit transfer completed successfully!');

        } catch (\Exception $e) {
            return back()->withErrors(['pin' => $e->getMessage()]);
        }
    }

    /**
     * Show transfer details.
     */
    public function showTransferDetails($transferId)
    {
        $transfer = \App\Models\CreditTransfer::findOrFail($transferId);
        $user = Auth::user();

        // Check if user is either sender or recipient
        if ($transfer->sender_id !== $user->id && $transfer->recipient_id !== $user->id) {
            abort(403);
        }

        return view('wallet.transfer-details', compact('transfer'));
    }

    /**
     * Show transfer history.
     */
    public function transferHistory()
    {
        $user = Auth::user();
        $transfers = $this->creditTransferService->getTransferHistory($user);
        $stats = $this->creditTransferService->getTransferStats($user);

        return view('wallet.transfer-history', compact('transfers', 'stats'));
    }
}