<?php

namespace App\Http\Middleware;

use App\Models\Level;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

/**
 * CheckLevel Middleware
 * 
 * Handles permission checking based on both the new role-based system
 * and legacy level-based system. Provides fallback logic to ensure
 * compatibility during the transition period.
 */
class CheckLevel
{
    /**
     * Handle an incoming request.
     * 
     * Validates user permissions for specific actions using a hybrid approach:
     * 1. First checks the new role-based permission system
     * 2. Falls back to the legacy level-based system if needed
     * 3. Enforces KYC requirements where applicable
     * 
     * @param Request $request The incoming request
     * @param Closure $next The next middleware in the pipeline
     * @param string $permission The permission to check (create_challenges, host_free_tournaments, etc.)
     * @return Response
     */
    public function handle(Request $request, Closure $next, string $permission): Response
    {
        $user = Auth::user();

        if (!$user) {
            return redirect()->route('login');
        }

        // Use assigned level first, fall back to calculated level
        $currentLevel = $user->currentLevel ?? $user->getCurrentLevel();

        if (!$currentLevel) {
            abort(403, 'Invalid user level.');
        }

        // Check specific permissions - using role-based system with level fallback
        switch ($permission) {
            case 'create_challenges':
                // Allow if user has proper role (basic_user, premium_user, creator, admin)
                // Check new role system first
                if (!in_array($user->role->value, ['basic_user', 'premium_user', 'creator', 'admin'])) {
                    // Check old level system as fallback
                    if (!$currentLevel || !$currentLevel->can_create_challenges) {
                        abort(403, 'You need to be at least Level 1 to create challenges.');
                    }
                } else {
                    // Also check if they have sufficient level even with correct role
                    if (!$currentLevel || !$currentLevel->can_create_challenges) {
                        abort(403, 'You need to be at least Level 1 to create challenges.');
                    }
                }
                break;

            case 'host_free_tournaments':
                // Check if user has proper role (creator, admin, premium_user) or level permissions
                $hasRolePermission = in_array($user->role->value, ['creator', 'admin', 'premium_user']);
                $hasLevelPermission = $currentLevel && $currentLevel->can_host_free_tournaments;
                
                if (!$hasRolePermission && !$hasLevelPermission) {
                    abort(403, 'You need to be highest or creator level to host tournaments.');
                }
                break;

            case 'host_paid_tournaments':
                // Check KYC first as it's required for both systems
                if ($user->kyc_status !== 'verified') {
                    abort(403, 'You need verified KYC status to host paid tournaments.');
                }
                
                // Check new level-based permission first
                if (!$user->canCreateTournamentsNewSystem()) {
                    // Fall back to role-based permission, then level system
                    if (!$user->canCreateTournaments()) {
                        if (!$currentLevel || !$currentLevel->can_host_paid_tournaments) {
                            abort(403, 'You need to be level 29+, Creator level, or Admin to host paid tournaments.');
                        }
                    }
                }
                break;

            case 'verified_kyc':
                // Enforce KYC verification requirement
                if ($user->kyc_status !== 'verified') {
                    abort(403, 'This action requires verified KYC status.');
                }
                break;

            default:
                // Invalid permission parameter - this indicates a developer error
                abort(500, 'Invalid permission check: ' . $permission);
        }

        return $next($request);
    }
}