<?php

namespace App\Listeners;

use App\Events\ChallengeAccepted;
use App\Models\Notification;
use App\Services\NotificationService;

class SendChallengeAcceptedNotification
{
    protected $notificationService;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * Handle the event.
     */
    public function handle(ChallengeAccepted $event): void
    {
        $challenge = $event->challenge;
        
        // Create database notification
        Notification::create([
            'user_id' => $challenge->creator_id,
            'type' => 'challenge_accepted',
            'title' => 'Challenge Accepted!',
            'body' => "{$challenge->accepter->username} accepted your challenge for {$challenge->game->name}.",
            'data' => [
                'challenge_id' => $challenge->id,
                'accepter_username' => $challenge->accepter->username,
            ],
        ]);

        // Send email notification if user has email
        if ($challenge->creator->email) {
            $this->notificationService->sendEmail(
                $challenge->creator->email,
                'Challenge Accepted - BattleMania',
                $this->getEmailTemplate($challenge),
                'BattleMania <noreply@battlemania.com>'
            );
        }

        // Send SMS notification if user has phone number
        if ($challenge->creator->phone) {
            $this->notificationService->sendSMS(
                $challenge->creator->phone,
                "🎮 BattleMania: {$challenge->accepter->username} accepted your {$challenge->game->name} challenge! Wager: ₦{$challenge->wager_amount}. Good luck!"
            );
        }
    }

    /**
     * Get email template for challenge accepted notification.
     */
    protected function getEmailTemplate($challenge): string
    {
        return "
        <html>
        <head>
            <style>
                body { font-family: Arial, sans-serif; background: #0a0a0a; color: #ffffff; margin: 0; padding: 20px; }
                .container { max-width: 600px; margin: 0 auto; background: #1a1a1a; border-radius: 10px; padding: 20px; }
                .header { text-align: center; margin-bottom: 30px; }
                .logo { font-size: 28px; font-weight: bold; color: #ff6b35; }
                .content { margin-bottom: 30px; }
                .challenge-details { background: #2a2a2a; padding: 20px; border-radius: 8px; margin: 20px 0; }
                .button { display: inline-block; background: #ff6b35; color: white; padding: 12px 24px; text-decoration: none; border-radius: 6px; font-weight: bold; }
                .footer { text-align: center; color: #888; font-size: 14px; margin-top: 30px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <div class='logo'>🎮 BattleMania</div>
                    <h1>Challenge Accepted!</h1>
                </div>
                <div class='content'>
                    <p>Great news! Your challenge has been accepted.</p>
                    
                    <div class='challenge-details'>
                        <h3>Challenge Details</h3>
                        <p><strong>Game:</strong> {$challenge->game->name}</p>
                        <p><strong>Accepted by:</strong> {$challenge->accepter->username}</p>
                        <p><strong>Wager Amount:</strong> ₦" . number_format($challenge->wager_amount, 2) . "</p>
                        <p><strong>Status:</strong> Ready to Play</p>
                    </div>
                    
                    <p>The match can now begin. Please coordinate with your opponent and submit proof when complete.</p>
                    
                    <p style='text-align: center;'>
                        <a href='" . route('challenges.show', $challenge->id) . "' class='button'>View Challenge</a>
                    </p>
                </div>
                <div class='footer'>
                    <p>BattleMania - Premium Gaming Challenges</p>
                    <p>This is an automated message. Please do not reply to this email.</p>
                </div>
            </div>
        </body>
        </html>
        ";
    }
}