<?php

namespace App\Listeners;

use App\Events\DisputeOpened;
use App\Models\Notification;
use App\Models\User;
use App\Services\RoleManagementService;
use App\Enums\UserRole;

class SendDisputeNotifications
{
    public function __construct(
        private RoleManagementService $roleService
    ) {}

    /**
     * Handle the event.
     */
    public function handle(DisputeOpened $event): void
    {
        $dispute = $event->dispute;
        $challenge = $dispute->challenge;
        
        // Auto-assign dispute to a moderator
        $this->roleService->assignDisputeToModerator($dispute);
        
        // Notify the other player in the challenge
        $otherUserId = $dispute->opened_by === $challenge->creator_id 
            ? $challenge->accepter_id 
            : $challenge->creator_id;
        
        if ($otherUserId) {
            Notification::create([
                'user_id' => $otherUserId,
                'type' => 'dispute_opened',
                'title' => 'Dispute Opened',
                'body' => "A dispute has been opened for your challenge in {$challenge->game->name}.",
                'data' => [
                    'challenge_id' => $challenge->id,
                    'dispute_id' => $dispute->id,
                ],
            ]);
        }

        // Notify assigned moderator if any
        if ($dispute->moderator_id) {
            Notification::create([
                'user_id' => $dispute->moderator_id,
                'type' => 'dispute_assigned',
                'title' => 'New Dispute Assigned',
                'body' => "A dispute has been assigned to you for review in {$challenge->game->name}.",
                'data' => [
                    'challenge_id' => $challenge->id,
                    'dispute_id' => $dispute->id,
                ],
            ]);
        }

        // Notify admins/organizers (they can see all disputes)
        $admins = User::where('is_organizer', true)->get();
        foreach ($admins as $admin) {
            Notification::create([
                'user_id' => $admin->id,
                'type' => 'admin_dispute_opened',
                'title' => 'New Dispute Requires Review',
                'body' => "A new dispute has been opened for a {$challenge->game->name} challenge.",
                'data' => [
                    'challenge_id' => $challenge->id,
                    'dispute_id' => $dispute->id,
                ],
            ]);
        }
    }
}