<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Challenge extends Model
{
    use HasFactory;

    protected $fillable = [
        'creator_id',
        'accepter_id',
        'game_id',
        'wager_amount',
        'best_of',
        'rules_text',
        'status',
        'result',
        'creator_game_username',
        'group_link',
        'group_id',
        'creator_claimed_result',
        'accepter_claimed_result',
    ];

    protected $casts = [
        'wager_amount' => 'decimal:2',
    ];

    /**
     * The creator of the challenge
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'creator_id');
    }

    /**
     * The accepter of the challenge
     */
    public function accepter(): BelongsTo
    {
        return $this->belongsTo(User::class, 'accepter_id');
    }

    /**
     * The game for this challenge
     */
    public function game(): BelongsTo
    {
        return $this->belongsTo(Game::class);
    }

    /**
     * Match proofs for this challenge
     */
    public function matchProofs(): HasMany
    {
        return $this->hasMany(MatchProof::class);
    }

    /**
     * Dispute for this challenge
     */
    public function dispute(): HasOne
    {
        return $this->hasOne(Dispute::class);
    }

    /**
     * Challenge invites
     */
    public function invites(): HasMany
    {
        return $this->hasMany(ChallengeInvite::class);
    }

    /**
     * Pending challenge invites
     */
    public function pendingInvites(): HasMany
    {
        return $this->hasMany(ChallengeInvite::class)->where('status', 'pending');
    }

    /**
     * Check if challenge can be accepted by a user
     */
    public function canBeAcceptedBy(User $user): bool
    {
        return $this->status === 'open' 
            && $this->creator_id !== $user->id
            && $user->wallet_balance >= $this->wager_amount;
    }

    /**
     * Check if challenge is completed
     */
    public function isCompleted(): bool
    {
        return $this->status === 'completed';
    }

    /**
     * Check if challenge is disputed
     */
    public function isDisputed(): bool
    {
        return $this->status === 'disputed';
    }

    /**
     * Get total escrow amount (both players' wagers)
     */
    public function getTotalEscrowAmount(): float
    {
        return $this->wager_amount * 2;
    }

    /**
     * Get winner of the challenge
     */
    public function getWinner(): ?User
    {
        if ($this->result === 'creator_win') {
            return $this->creator;
        } elseif ($this->result === 'accepter_win') {
            return $this->accepter;
        }
        
        return null;
    }

    /**
     * Invite a user to the challenge
     */
    public function inviteUser(User $invitee, User $inviter, string $message = null): ?ChallengeInvite
    {
        // Check if challenge is still open
        if ($this->status !== 'open') {
            return null;
        }

        // Check if user is not the creator
        if ($this->creator_id === $invitee->id) {
            return null;
        }

        // Check if user already has an invite for this challenge
        if ($this->hasUserInvite($invitee)) {
            return null;
        }

        return $this->invites()->create([
            'inviter_id' => $inviter->id,
            'invitee_id' => $invitee->id,
            'message' => $message,
            'expires_at' => now()->addDays(3), // Challenge invites expire in 3 days
        ]);
    }

    /**
     * Check if user has invite for this challenge
     */
    public function hasUserInvite(User $user): bool
    {
        return $this->invites()->where('invitee_id', $user->id)
                   ->whereIn('status', ['pending', 'accepted'])
                   ->exists();
    }

    /**
     * Check if both players have submitted their result claims
     */
    public function hasBothClaimedResults(): bool
    {
        return $this->creator_claimed_result !== 'none' && 
               $this->accepter_claimed_result !== 'none';
    }

    /**
     * Check if results match (one claims win, other claims loss)
     */
    public function resultsMatch(): bool
    {
        return ($this->creator_claimed_result === 'win' && $this->accepter_claimed_result === 'loss') ||
               ($this->creator_claimed_result === 'loss' && $this->accepter_claimed_result === 'win');
    }

    /**
     * Check if both players claim victory (dispute scenario)
     */
    public function bothClaimVictory(): bool
    {
        return $this->creator_claimed_result === 'win' && $this->accepter_claimed_result === 'win';
    }

    /**
     * Check if both players claim loss (forfeit scenario)
     */
    public function bothClaimLoss(): bool
    {
        return $this->creator_claimed_result === 'loss' && $this->accepter_claimed_result === 'loss';
    }

    /**
     * Check if one player has claimed loss (for immediate resolution)
     */
    public function hasPlayerClaimedLoss(): bool
    {
        return $this->creator_claimed_result === 'loss' || $this->accepter_claimed_result === 'loss';
    }

    /**
     * Get the winner when one player claims loss (immediate resolution)
     */
    public function getWinnerFromLossClaimant(): ?User
    {
        if ($this->creator_claimed_result === 'loss') {
            return $this->accepter;
        }
        
        if ($this->accepter_claimed_result === 'loss') {
            return $this->creator;
        }
        
        return null;
    }

    /**
     * Get the winner based on matching result claims
     */
    public function getWinnerFromClaims(): ?User
    {
        if (!$this->resultsMatch()) {
            return null;
        }

        if ($this->creator_claimed_result === 'win') {
            return $this->creator;
        }

        if ($this->accepter_claimed_result === 'win') {
            return $this->accepter;
        }

        return null;
    }

    /**
     * Reset result claims (useful for dispute resolution)
     */
    public function resetResultClaims(): void
    {
        $this->update([
            'creator_claimed_result' => 'none',
            'accepter_claimed_result' => 'none',
        ]);
    }
}