<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ChallengeInvite extends Model
{
    use HasFactory;

    protected $fillable = [
        'challenge_id',
        'inviter_id',
        'invitee_id',
        'status',
        'message',
        'expires_at',
        'responded_at',
    ];

    protected $casts = [
        'expires_at' => 'datetime',
        'responded_at' => 'datetime',
    ];

    /**
     * Get the challenge this invite belongs to
     */
    public function challenge(): BelongsTo
    {
        return $this->belongsTo(Challenge::class);
    }

    /**
     * Get the user who sent the invite
     */
    public function inviter(): BelongsTo
    {
        return $this->belongsTo(User::class, 'inviter_id');
    }

    /**
     * Get the user who received the invite
     */
    public function invitee(): BelongsTo
    {
        return $this->belongsTo(User::class, 'invitee_id');
    }

    /**
     * Check if invite is pending
     */
    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    /**
     * Check if invite is accepted
     */
    public function isAccepted(): bool
    {
        return $this->status === 'accepted';
    }

    /**
     * Check if invite is declined
     */
    public function isDeclined(): bool
    {
        return $this->status === 'declined';
    }

    /**
     * Check if invite is expired
     */
    public function isExpired(): bool
    {
        return $this->expires_at && $this->expires_at->isPast();
    }

    /**
     * Check if invite can be responded to
     */
    public function canBeResponded(): bool
    {
        return $this->isPending() && !$this->isExpired();
    }

    /**
     * Accept the invite
     */
    public function accept(): bool
    {
        if (!$this->canBeResponded()) {
            return false;
        }

        // Check if challenge is still open
        if ($this->challenge->status !== 'open') {
            return false;
        }

        $this->update([
            'status' => 'accepted',
            'responded_at' => now(),
        ]);

        return true;
    }

    /**
     * Decline the invite
     */
    public function decline(): bool
    {
        if (!$this->canBeResponded()) {
            return false;
        }

        $this->update([
            'status' => 'declined',
            'responded_at' => now(),
        ]);

        return true;
    }
}
