<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Dispute extends Model
{
    use HasFactory;

    protected $fillable = [
        'challenge_id',
        'opened_by',
        'reason',
        'status',
        'resolution',
        'admin_id',
        'moderator_id',
        'assigned_to_moderator_at',
        'creator_evidence_path',
        'accepter_evidence_path',
        'admin_notified',
    ];

    protected $casts = [
        'admin_notified' => 'boolean',
        'assigned_to_moderator_at' => 'datetime',
    ];

    /**
     * The challenge this dispute is about
     */
    public function challenge(): BelongsTo
    {
        return $this->belongsTo(Challenge::class);
    }

    /**
     * The user who opened the dispute
     */
    public function openedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'opened_by');
    }

    /**
     * The admin who resolved the dispute
     */
    public function admin(): BelongsTo
    {
        return $this->belongsTo(User::class, 'admin_id');
    }

    /**
     * The moderator assigned to this dispute
     */
    public function moderator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'moderator_id');
    }

    /**
     * Check if dispute is open
     */
    public function isOpen(): bool
    {
        return $this->status === 'open';
    }

    /**
     * Check if dispute is resolved
     */
    public function isResolved(): bool
    {
        return $this->status === 'resolved';
    }

    /**
     * Check if creator has uploaded evidence
     */
    public function hasCreatorEvidence(): bool
    {
        return !empty($this->creator_evidence_path);
    }

    /**
     * Check if accepter has uploaded evidence
     */
    public function hasAccepterEvidence(): bool
    {
        return !empty($this->accepter_evidence_path);
    }

    /**
     * Check if both parties have uploaded evidence
     */
    public function hasBothEvidence(): bool
    {
        return $this->hasCreatorEvidence() && $this->hasAccepterEvidence();
    }

    /**
     * Mark as admin notified
     */
    public function markAdminNotified(): void
    {
        $this->update(['admin_notified' => true]);
    }

    /**
     * Assign dispute to a moderator
     */
    public function assignToModerator(User $moderator): void
    {
        $this->update([
            'moderator_id' => $moderator->id,
            'assigned_to_moderator_at' => now(),
        ]);
    }

    /**
     * Check if dispute is assigned to a moderator
     */
    public function hasAssignedModerator(): bool
    {
        return !is_null($this->moderator_id);
    }

    /**
     * Auto-assign to random available moderator
     */
    public static function autoAssignToModerator(Dispute $dispute): void
    {
        // Don't reassign if already assigned
        if ($dispute->hasAssignedModerator()) {
            return;
        }

        // Get available moderators with lowest dispute load
        $moderators = User::where('role', 'moderator')
                         ->where('status', 'active')
                         ->withCount(['moderatedDisputes' => function ($query) {
                             $query->where('status', 'open');
                         }])
                         ->orderBy('moderated_disputes_count', 'asc')
                         ->get();

        if ($moderators->isNotEmpty()) {
            // Pick a random moderator from those with the lowest load
            $lowestLoad = $moderators->first()->moderated_disputes_count;
            $availableModerators = $moderators->where('moderated_disputes_count', $lowestLoad);
            
            $selectedModerator = $availableModerators->random();
            $dispute->assignToModerator($selectedModerator);
        }
    }
}