<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Friend extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'friend_id',
        'status',
        'message',
        'request_sent_at',
        'responded_at',
    ];

    protected $casts = [
        'request_sent_at' => 'datetime',
        'responded_at' => 'datetime',
    ];

    /**
     * The user who sent the friend request
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * The user who received the friend request
     */
    public function friend(): BelongsTo
    {
        return $this->belongsTo(User::class, 'friend_id');
    }

    /**
     * Check if friend request is pending
     */
    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    /**
     * Check if friend request is accepted
     */
    public function isAccepted(): bool
    {
        return $this->status === 'accepted';
    }

    /**
     * Check if friend request is declined
     */
    public function isDeclined(): bool
    {
        return $this->status === 'declined';
    }

    /**
     * Check if friend is blocked
     */
    public function isBlocked(): bool
    {
        return $this->status === 'blocked';
    }

    /**
     * Accept the friend request
     */
    public function accept(): bool
    {
        if (!$this->isPending()) {
            return false;
        }

        $this->update([
            'status' => 'accepted',
            'responded_at' => now(),
        ]);

        // Create reciprocal friendship
        self::updateOrCreate([
            'user_id' => $this->friend_id,
            'friend_id' => $this->user_id,
        ], [
            'status' => 'accepted',
            'request_sent_at' => $this->request_sent_at,
            'responded_at' => now(),
        ]);

        return true;
    }

    /**
     * Decline the friend request
     */
    public function decline(): bool
    {
        if (!$this->isPending()) {
            return false;
        }

        $this->update([
            'status' => 'declined',
            'responded_at' => now(),
        ]);

        return true;
    }

    /**
     * Block the user
     */
    public function block(): bool
    {
        $this->update([
            'status' => 'blocked',
            'responded_at' => now(),
        ]);

        // Also block from the other side
        self::updateOrCreate([
            'user_id' => $this->friend_id,
            'friend_id' => $this->user_id,
        ], [
            'status' => 'blocked',
            'request_sent_at' => $this->request_sent_at,
            'responded_at' => now(),
        ]);

        return true;
    }

    /**
     * Get mutual friends count between two users
     */
    public static function getMutualFriendsCount(User $user1, User $user2): int
    {
        $user1Friends = self::where('user_id', $user1->id)
                           ->where('status', 'accepted')
                           ->pluck('friend_id');

        $user2Friends = self::where('user_id', $user2->id)
                           ->where('status', 'accepted')
                           ->pluck('friend_id');

        return $user1Friends->intersect($user2Friends)->count();
    }
}
