<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Casts\Attribute;

class NotificationProvider extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'type',
        'display_name',
        'configuration',
        'is_active',
        'is_default',
        'priority',
        'test_results',
        'last_tested_at',
    ];

    protected $casts = [
        'configuration' => 'array',
        'test_results' => 'array',
        'is_active' => 'boolean',
        'is_default' => 'boolean',
        'last_tested_at' => 'datetime',
    ];

    /**
     * Get active providers for a specific type.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get providers by type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Get providers ordered by priority.
     */
    public function scopeOrderedByPriority($query)
    {
        return $query->orderBy('priority', 'asc');
    }

    /**
     * Get the default provider for a type.
     */
    public static function getDefault($type)
    {
        return static::byType($type)->where('is_default', true)->active()->first();
    }

    /**
     * Get active providers for a type ordered by priority.
     */
    public static function getActiveProviders($type)
    {
        return static::byType($type)->active()->orderedByPriority()->get();
    }

    /**
     * Check if provider configuration is valid.
     */
    public function isConfigurationValid(): bool
    {
        $requiredFields = $this->getRequiredConfigurationFields();
        
        foreach ($requiredFields as $field) {
            if (!isset($this->configuration[$field]) || empty($this->configuration[$field])) {
                return false;
            }
        }

        return true;
    }

    /**
     * Get required configuration fields for the provider.
     */
    public function getRequiredConfigurationFields(): array
    {
        return match($this->name) {
            'resend' => ['api_key'],
            'mailgun' => ['domain', 'secret', 'endpoint'],
            'sendgrid' => ['api_key'],
            'termii' => ['api_key', 'sender_id'],
            'twilio' => ['account_sid', 'auth_token', 'from_number'],
            'nexmo' => ['api_key', 'api_secret', 'from_number'],
            default => [],
        };
    }

    /**
     * Get configuration field labels for the admin panel.
     */
    public function getConfigurationFieldLabels(): array
    {
        return match($this->name) {
            'resend' => [
                'api_key' => 'API Key',
            ],
            'mailgun' => [
                'domain' => 'Domain',
                'secret' => 'API Key',
                'endpoint' => 'Endpoint (e.g., api.mailgun.net)',
            ],
            'sendgrid' => [
                'api_key' => 'API Key',
            ],
            'termii' => [
                'api_key' => 'API Key',
                'sender_id' => 'Sender ID',
            ],
            'twilio' => [
                'account_sid' => 'Account SID',
                'auth_token' => 'Auth Token',
                'from_number' => 'From Number',
            ],
            'nexmo' => [
                'api_key' => 'API Key',
                'api_secret' => 'API Secret',
                'from_number' => 'From Number',
            ],
            default => [],
        };
    }
}
